(function() {
    'use strict';
    
    // 直接使用全局变量，避免参数传递问题
    const { registerBlockType } = wp.blocks;
    const { InspectorControls } = wp.blockEditor;
    const { PanelBody, TextControl, Button } = wp.components;
    const { __ } = wp.i18n;

    // 检查 MediaUpload 是否可用
    const MediaUpload = wp.blockEditor?.MediaUpload || wp.editor?.MediaUpload;

    // 极简区块定义
    registerBlockType('ncewpv/paid-video', {
        title: __('收费视频', 'ncewpv'),
        description: __('添加收费视频区块', 'ncewpv'),
        icon: 'video-alt3',
        category: 'media',
        
        attributes: {
            freeVideoUrl: {
                type: 'string',
                default: ''
            },
            paidVideoUrl: {
                type: 'string',
                default: ''
            },
            price: {
                type: 'number',
                default: 0
            },
            previewLength: {
                type: 'number',
                default: 30
            }
        },
        
        edit: function(props) {
            const { attributes, setAttributes } = props;
            
            // 确保所有属性都有值
            const freeVideoUrl = attributes.freeVideoUrl || '';
            const paidVideoUrl = attributes.paidVideoUrl || '';
            const price = attributes.price || 0;
            const previewLength = attributes.previewLength || 30;
            
            // 简单的更新函数
            const updateFreeVideo = (value) => {
                setAttributes({ freeVideoUrl: value });
            };
            
            const updatePaidVideo = (value) => {
                setAttributes({ paidVideoUrl: value });
            };
            
            const updatePrice = (value) => {
                setAttributes({ price: parseFloat(value) || 0 });
            };
            
            const updatePreviewLength = (value) => {
                setAttributes({ previewLength: parseInt(value) || 30 });
            };

            // 媒体选择处理函数
            const onSelectFreeVideo = (media) => {
                if (media && media.url) {
                    // 检查文件大小
                    if (media.filesizeInBytes > 100 * 1024 * 1024) { // 100MB
                        alert('警告：视频文件大小超过100MB，可能会影响加载速度。建议压缩视频文件。');
                    }
                    updateFreeVideo(media.url);
                }
            };

            const onSelectPaidVideo = (media) => {
                if (media && media.url) {
                    // 检查文件大小
                    if (media.filesizeInBytes > 100 * 1024 * 1024) { // 100MB
                        alert('警告：视频文件大小超过100MB，可能会影响加载速度。建议压缩视频文件。');
                    }
                    updatePaidVideo(media.url);
                }
            };
            
            return wp.element.createElement('div', { className: 'ncewpv-block-editor' },
                wp.element.createElement(InspectorControls, {},
                    wp.element.createElement(PanelBody, { 
                        title: __('视频设置', 'ncewpv'),
                        initialOpen: true 
                    },
                        // 免费视频设置
                        wp.element.createElement('div', { style: { marginBottom: '20px' } },
                            wp.element.createElement('p', { 
                                style: { 
                                    marginBottom: '8px', 
                                    fontWeight: 'bold',
                                    fontSize: '13px'
                                } 
                            }, __('免费预览视频', 'ncewpv')),
                            
                            // 文件大小限制提示
                            wp.element.createElement('div', { className: 'ncewpv-size-warning' },
                                __('建议视频大小不超过100MB，格式为MP4', 'ncewpv')
                            ),
                            
                            // 文本输入框
                            wp.element.createElement(TextControl, {
                                value: freeVideoUrl,
                                onChange: updateFreeVideo,
                                placeholder: __('输入视频URL或使用媒体库选择', 'ncewpv'),
                                style: { marginBottom: '10px' }
                            }),
                            
                            // 媒体库按钮（如果可用）
                            MediaUpload && wp.element.createElement(MediaUpload, {
                                onSelect: onSelectFreeVideo,
                                allowedTypes: ['video'],
                                value: freeVideoUrl,
                                render: function(obj) {
                                    return wp.element.createElement(Button, {
                                        isSecondary: true,
                                        onClick: obj.open,
                                        style: { marginBottom: '10px' }
                                    }, freeVideoUrl ? __('更换视频', 'ncewpv') : __('从媒体库选择', 'ncewpv'));
                                }
                            }),
                            
                            // 视频预览
                            freeVideoUrl && wp.element.createElement('div', { style: { marginTop: '10px' } },
                                wp.element.createElement('video', {
                                    src: freeVideoUrl,
                                    controls: true,
                                    style: {
                                        width: '100%',
                                        maxWidth: '300px',
                                        maxHeight: '200px',
                                        borderRadius: '4px'
                                    }
                                })
                            )
                        ),
                        
                        // 完整视频设置
                        wp.element.createElement('div', { style: { marginBottom: '20px' } },
                            wp.element.createElement('p', { 
                                style: { 
                                    marginBottom: '8px', 
                                    fontWeight: 'bold',
                                    fontSize: '13px'
                                } 
                            }, __('完整收费视频', 'ncewpv')),
                            
                            // 文件大小限制提示
                            wp.element.createElement('div', { className: 'ncewpv-size-warning' },
                                __('建议视频大小不超过100MB，格式为MP4', 'ncewpv')
                            ),
                            
                            // 文本输入框
                            wp.element.createElement(TextControl, {
                                value: paidVideoUrl,
                                onChange: updatePaidVideo,
                                placeholder: __('输入视频URL或使用媒体库选择', 'ncewpv'),
                                style: { marginBottom: '10px' }
                            }),
                            
                            // 媒体库按钮（如果可用）
                            MediaUpload && wp.element.createElement(MediaUpload, {
                                onSelect: onSelectPaidVideo,
                                allowedTypes: ['video'],
                                value: paidVideoUrl,
                                render: function(obj) {
                                    return wp.element.createElement(Button, {
                                        isSecondary: true,
                                        onClick: obj.open,
                                        style: { marginBottom: '10px' }
                                    }, paidVideoUrl ? __('更换视频', 'ncewpv') : __('从媒体库选择', 'ncewpv'));
                                }
                            }),
                            
                            // 视频预览
                            paidVideoUrl && wp.element.createElement('div', { style: { marginTop: '10px' } },
                                wp.element.createElement('video', {
                                    src: paidVideoUrl,
                                    controls: true,
                                    style: {
                                        width: '100%',
                                        maxWidth: '300px',
                                        maxHeight: '200px',
                                        borderRadius: '4px'
                                    }
                                })
                            )
                        ),
                        
                        // 价格设置
                        wp.element.createElement(TextControl, {
                            label: __('价格 (¥)', 'ncewpv'),
                            type: 'number',
                            value: price,
                            onChange: updatePrice,
                            help: __('设置视频价格', 'ncewpv'),
                            style: { marginTop: '20px' }
                        }),
                        
                        // 预览时长设置
                        wp.element.createElement(TextControl, {
                            label: __('预览时长 (秒)', 'ncewpv'),
                            type: 'number',
                            value: previewLength,
                            onChange: updatePreviewLength,
                            help: __('设置免费预览时长', 'ncewpv'),
                            style: { marginTop: '10px' }
                        })
                    )
                ),
                
                wp.element.createElement('div', { className: 'ncewpv-block-preview' },
                    wp.element.createElement('h4', {}, __('收费视频区块预览', 'ncewpv')),
                    
                    wp.element.createElement('div', { className: 'ncewpv-preview-info' },
                        wp.element.createElement('p', {}, 
                            wp.element.createElement('strong', {}, __('免费视频:', 'ncewpv')),
                            ' ',
                            freeVideoUrl ? __('已设置', 'ncewpv') : __('未设置', 'ncewpv')
                        ),
                        
                        wp.element.createElement('p', {}, 
                            wp.element.createElement('strong', {}, __('完整视频:', 'ncewpv')),
                            ' ',
                            paidVideoUrl ? __('已设置', 'ncewpv') : __('未设置', 'ncewpv')
                        ),
                        
                        wp.element.createElement('p', {}, 
                            wp.element.createElement('strong', {}, __('价格:', 'ncewpv')),
                            ' ¥',
                            price
                        ),
                        
                        wp.element.createElement('p', {}, 
                            wp.element.createElement('strong', {}, __('预览时长:', 'ncewpv')),
                            ' ',
                            previewLength,
                            __('秒', 'ncewpv')
                        )
                    ),
                    
                    (!freeVideoUrl || !paidVideoUrl) && 
                        wp.element.createElement('div', { className: 'ncewpv-config-warning' },
                            __('请设置免费视频和完整视频URL', 'ncewpv')
                        )
                )
            );
        },
        
        save: function() {
            return null;
        }
    });
})();