(function($) {
    'use strict';
    
    console.log('NCEWPV: frontend.js 开始加载');
    
    // 全局变量
    var ncewpv = {
        debug: true,
        settings: {},
        init: function() {
            this.settings = window.ncewpv_ajax.settings || {};
            this.bindEvents();
            this.log('前端初始化完成');
        },
        
        bindEvents: function() {
            var self = this;
            
            // 使用事件委托绑定购买按钮点击事件
            $(document).on('click', '.ncewpv-purchase-btn', function(e) {
                e.preventDefault();
                e.stopPropagation();
                self.handlePurchaseClick($(this), e);
            });
            
            // 绑定关闭按钮事件
            $(document).on('click', '.ncewpv-modal-close, .ncewpv-modal-overlay', function(e) {
                if (e.target === this || $(e.target).hasClass('ncewpv-modal-close')) {
                    $('.ncewpv-payment-modal').hide();
                    $('body').css('overflow', '');
                }
            });
            
            // ESC键关闭
            $(document).on('keyup', function(e) {
                if (e.keyCode === 27) {
                    $('.ncewpv-payment-modal').hide();
                    $('body').css('overflow', '');
                }
            });
            
            this.log('事件绑定完成');
        },
        
        handlePurchaseClick: function($button, e) {
            var $container = $button.closest('.ncewpv-video-container');
            var postId = $container.data('post-id');
            
            this.log('购买按钮点击，文章ID:', postId);
            
            if (postId) {
                this.showPaymentModal(postId);
            } else {
                this.error('无法获取文章ID');
                alert('购买功能暂时不可用，请刷新页面重试');
            }
        },
        
        showPaymentModal: function(postId) {
            this.log('显示支付模态框，文章ID:', postId);
            
            // 先隐藏所有其他模态框
            $('.ncewpv-payment-modal').hide();
            
            // 查找或创建模态框
            var $modal = $('.ncewpv-payment-modal[data-post-id="' + postId + '"]');
            
            if ($modal.length === 0) {
                this.log('创建新的支付模态框');
                $modal = this.createPaymentModal(postId);
            }
            
            // 显示模态框
            $modal.show();
            $('body').css('overflow', 'hidden');
            
            this.log('模态框显示成功');
        },
        
        createPaymentModal: function(postId) {
            this.log('创建支付模态框');
            
            // 获取价格信息
            var $container = $('.ncewpv-video-container[data-post-id="' + postId + '"]');
            var attributes = $container.data('attributes') || {};
            var finalPrice = this.calculateFinalPrice(attributes.price || 0);
            
            var modalHTML = `
                <div class="ncewpv-payment-modal" data-post-id="${postId}">
                    <div class="ncewpv-modal-overlay"></div>
                    <div class="ncewpv-modal-content">
                        <div class="ncewpv-modal-header">
                            <h3>购买完整视频</h3>
                            <button class="ncewpv-modal-close">&times;</button>
                        </div>
                        <div class="ncewpv-modal-body">
                            <div class="ncewpv-payment-info">
                                <h4>${attributes.title || '视频内容'}</h4>
                                <div class="ncewpv-price-breakdown">
                                    <div class="price-item">
                                        <span>视频价格:</span>
                                        <span>${(attributes.price || 0).toFixed(2)} 元</span>
                                    </div>
                                    <div class="price-item">
                                        <span>手续费:</span>
                                        <span>${(finalPrice - (attributes.price || 0)).toFixed(2)} 元</span>
                                    </div>
                                    <div class="price-item total">
                                        <span>总计:</span>
                                        <span class="ncewpv-total-price">${finalPrice.toFixed(2)} 元</span>
                                    </div>
                                </div>
                            </div>
                            <div class="ncewpv-paypal-container">
                                <div id="ncewpv-paypal-button-container-${postId}"></div>
                            </div>
                            <div class="ncewpv-payment-status"></div>
                        </div>
                    </div>
                </div>
            `;
            
            $('body').append(modalHTML);
            var $modal = $('.ncewpv-payment-modal[data-post-id="' + postId + '"]');
            
            // 初始化PayPal按钮
            this.initPayPalButtons(postId, finalPrice);
            
            return $modal;
        },
        
        initPayPalButtons: function(postId, amount) {
            this.log('初始化PayPal按钮，金额:', amount);
            
            var $status = $('.ncewpv-payment-modal[data-post-id="' + postId + '"] .ncewpv-payment-status');
            var $container = $('#ncewpv-paypal-button-container-' + postId);
            
            // 检查PayPal SDK
            if (typeof paypal === 'undefined') {
                this.error('PayPal SDK未加载');
                $container.html('<div class="error-message">支付系统未就绪，请刷新页面</div>');
                return;
            }
            
            // 检查PayPal配置
            if (!this.settings.client_id || !this.settings.client_secret) {
                this.warn('PayPal未配置');
                $container.html('<div class="config-message">请先在插件设置中配置PayPal API凭证</div>');
                return;
            }
            
            this.log('PayPal配置正常，创建按钮');
            
            try {
                paypal.Buttons({
                    style: {
                        shape: 'rect',
                        color: 'gold',
                        layout: 'vertical',
                        label: 'paypal',
                        height: 45
                    },
                    
                    createOrder: function(data, actions) {
                        return ncewpv.createPayPalOrder(postId, amount);
                    },
                    
                    onApprove: function(data, actions) {
                        return ncewpv.capturePayPalOrder(data.orderID, postId);
                    },
                    
                    onError: function(err) {
                        ncewpv.error('PayPal错误:', err);
                        $status.html('<div class="error">支付过程中发生错误，请重试</div>').show();
                    },
                    
                    onCancel: function(data) {
                        ncewpv.log('用户取消支付');
                        $status.hide();
                    }
                    
                }).render('#ncewpv-paypal-button-container-' + postId).then(function(instance) {
                    ncewpv.log('PayPal按钮渲染成功');
                }).catch(function(err) {
                    ncewpv.error('PayPal按钮渲染失败:', err);
                    $container.html('<div class="error-message">支付按钮加载失败: ' + err.message + '</div>');
                });
            } catch (error) {
                this.error('PayPal按钮初始化异常:', error);
                $container.html('<div class="error-message">支付系统初始化失败</div>');
            }
        },
        
        createPayPalOrder: function(postId, amount) {
            this.log('创建PayPal订单请求');
            return new Promise(function(resolve, reject) {
                $.ajax({
                    url: ncewpv_ajax.ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'ncewpv_create_paypal_order',
                        post_id: postId,
                        amount: amount,
                        nonce: ncewpv_ajax.nonce
                    },
                    success: function(response) {
                        ncewpv.log('创建订单响应:', response);
                        if (response.success) {
                            resolve(response.data.id);
                        } else {
                            reject(response.data);
                        }
                    },
                    error: function(xhr, status, error) {
                        ncewpv.error('创建订单请求失败:', error);
                        reject('创建订单失败: ' + error);
                    }
                });
            });
        },
        
        capturePayPalOrder: function(orderID, postId) {
            this.log('捕获PayPal订单:', orderID);
            var $status = $('.ncewpv-payment-modal[data-post-id="' + postId + '"] .ncewpv-payment-status');
            
            $status.html('<div class="processing"><span class="ncewpv-loading"></span>支付处理中...</div>').show();
            
            return new Promise(function(resolve, reject) {
                $.ajax({
                    url: ncewpv_ajax.ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'ncewpv_capture_paypal_order',
                        order_id: orderID,
                        post_id: postId,
                        nonce: ncewpv_ajax.nonce
                    },
                    success: function(response) {
                        ncewpv.log('捕获订单响应:', response);
                        if (response.success) {
                            $status.html('<div class="success">支付成功！正在解锁完整视频...</div>').show();
                            
                            setTimeout(function() {
                                ncewpv.unlockFullVideo(postId);
                                $('.ncewpv-payment-modal[data-post-id="' + postId + '"]').hide();
                                $('body').css('overflow', '');
                            }, 2000);
                            
                            resolve(response.data);
                        } else {
                            $status.html('<div class="error">支付验证失败: ' + response.data + '</div>').show();
                            reject(response.data);
                        }
                    },
                    error: function(xhr, status, error) {
                        ncewpv.error('捕获订单请求失败:', error);
                        $status.html('<div class="error">网络错误: ' + error + '</div>').show();
                        reject(error);
                    }
                });
            });
        },
        
        unlockFullVideo: function(postId) {
            this.log('解锁完整视频');
            var $container = $('.ncewpv-video-container[data-post-id="' + postId + '"]');
            var attributes = $container.data('attributes') || {};
            
            if (attributes.paidVideoUrl) {
                $container.find('.ncewpv-preview-section').replaceWith(`
                    <div class="ncewpv-full-video">
                        <div class="ncewpv-video-wrapper">
                            <video controls playsinline autoplay>
                                <source src="${attributes.paidVideoUrl}" type="video/mp4">
                                您的浏览器不支持视频播放
                            </video>
                        </div>
                        <div class="ncewpv-purchase-success">
                            <h4>已购买完整视频</h4>
                            <p>您可以无限次观看此视频</p>
                        </div>
                    </div>
                `);
                
                $container.attr('data-has-purchased', '1');
                this.log('视频解锁成功');
            }
        },
        
        calculateFinalPrice: function(originalPrice) {
            var feeRate = this.settings.fee_rate || 1.03;
            var fixedFee = this.settings.fixed_fee || 0;
            
            var finalPrice = (originalPrice * feeRate) + fixedFee;
            return Math.round(finalPrice * 100) / 100;
        },
        
        log: function() {
            if (this.debug) {
                console.log.apply(console, ['NCEWPV:'].concat(Array.prototype.slice.call(arguments)));
            }
        },
        
        warn: function() {
            console.warn.apply(console, ['NCEWPV:'].concat(Array.prototype.slice.call(arguments)));
        },
        
        error: function() {
            console.error.apply(console, ['NCEWPV:'].concat(Array.prototype.slice.call(arguments)));
        }
    };
    
    // 文档加载完成后初始化
    $(document).ready(function() {
        ncewpv.init();
    });
    
})(jQuery);