<?php
/**
 * 日志管理页面
 */

class NCEWPV_Admin_Logs {
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
    }
    
    public function add_admin_menu() {
        add_submenu_page(
            'options-general.php',
            '系统日志',
            '系统日志',
            'manage_options',
            'ncewpv-logs',
            array($this, 'logs_page')
        );
    }
    
    public function enqueue_admin_scripts($hook) {
        if ($hook !== 'settings_page_ncewpv-logs') {
            return;
        }
        
        wp_enqueue_style(
            'ncewpv-admin-logs',
            NCEWPV_PLUGIN_URL . 'assets/css/admin-logs.css',
            array(),
            NCEWPV_PLUGIN_VERSION
        );
        
        wp_enqueue_script(
            'ncewpv-admin-logs',
            NCEWPV_PLUGIN_URL . 'assets/js/admin-logs.js',
            array('jquery'),
            NCEWPV_PLUGIN_VERSION,
            true
        );
        
        wp_localize_script('ncewpv-admin-logs', 'ncewpv_logs', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('ncewpv_logs_nonce'),
            'confirm_clear' => '确定要清理日志吗？此操作不可恢复。'
        ));
    }
    
    public function logs_page() {
        $error_handler = NCEWPV_Error_Handler::get_instance();
        $stats = $error_handler->get_log_stats();
        
        // 处理筛选和分页
        $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
        $per_page = 50;
        $offset = ($current_page - 1) * $per_page;
        
        $args = array(
            'level' => isset($_GET['level']) ? sanitize_text_field($_GET['level']) : '',
            'user_id' => isset($_GET['user_id']) ? intval($_GET['user_id']) : '',
            'date_from' => isset($_GET['date_from']) ? sanitize_text_field($_GET['date_from']) : '',
            'date_to' => isset($_GET['date_to']) ? sanitize_text_field($_GET['date_to']) : '',
            'search' => isset($_GET['s']) ? sanitize_text_field($_GET['s']) : '',
            'limit' => $per_page,
            'offset' => $offset
        );
        
        $logs = $error_handler->get_logs($args);
        $total_logs = $stats->total;
        $total_pages = ceil($total_logs / $per_page);
        
        ?>
        <div class="wrap ncewpv-logs-wrap">
            <h1 class="wp-heading-inline">系统日志</h1>
            
            <?php $this->display_stats($stats); ?>
            <?php $this->display_filters($args); ?>
            <?php $this->display_toolbar(); ?>
            
            <div class="ncewpv-logs-container">
                <?php if (empty($logs)): ?>
                    <div class="notice notice-info">
                        <p>没有找到日志记录</p>
                    </div>
                <?php else: ?>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th width="80">级别</th>
                                <th width="120">时间</th>
                                <th width="100">用户</th>
                                <th width="120">IP地址</th>
                                <th>消息</th>
                                <th width="80">操作</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($logs as $log): ?>
                                <tr class="log-level-<?php echo strtolower($log->level); ?>">
                                    <td>
                                        <span class="ncewpv-log-level ncewpv-log-level-<?php echo strtolower($log->level); ?>">
                                            <?php echo esc_html($log->level); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <abbr title="<?php echo esc_attr($log->created_at); ?>">
                                            <?php echo human_time_diff(strtotime($log->created_at), current_time('timestamp')) . '前'; ?>
                                        </abbr>
                                    </td>
                                    <td>
                                        <?php if ($log->user_id): ?>
                                            <?php 
                                            $user = get_userdata($log->user_id);
                                            echo $user ? esc_html($user->display_name) : '用户已删除';
                                            ?>
                                        <?php else: ?>
                                            <em>系统</em>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo esc_html($log->ip_address); ?></td>
                                    <td>
                                        <div class="log-message">
                                            <?php echo esc_html($log->message); ?>
                                        </div>
                                        <?php if ($log->context): ?>
                                            <div class="log-context" style="display: none;">
                                                <pre><?php echo esc_html($log->context); ?></pre>
                                            </div>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if ($log->context): ?>
                                            <button type="button" class="button button-small view-context" 
                                                    data-context="<?php echo esc_attr($log->context); ?>">
                                                详情
                                            </button>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    
                    <?php $this->display_pagination($current_page, $total_pages); ?>
                    
                <?php endif; ?>
            </div>
            
            <!-- 上下文详情模态框 -->
            <div id="ncewpv-log-context-modal" class="ncewpv-modal" style="display: none;">
                <div class="ncewpv-modal-overlay"></div>
                <div class="ncewpv-modal-content">
                    <div class="ncewpv-modal-header">
                        <h3>日志详情</h3>
                        <button class="ncewpv-modal-close">&times;</button>
                    </div>
                    <div class="ncewpv-modal-body">
                        <pre id="ncewpv-log-context-content"></pre>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * 显示统计信息
     */
    private function display_stats($stats) {
        ?>
        <div class="ncewpv-logs-stats">
            <div class="stat-card">
                <div class="stat-number"><?php echo esc_html($stats->total); ?></div>
                <div class="stat-label">总日志数</div>
            </div>
            <div class="stat-card error">
                <div class="stat-number"><?php echo esc_html($stats->critical + $stats->errors); ?></div>
                <div class="stat-label">错误/严重</div>
            </div>
            <div class="stat-card warning">
                <div class="stat-number"><?php echo esc_html($stats->warnings); ?></div>
                <div class="stat-label">警告</div>
            </div>
            <div class="stat-card info">
                <div class="stat-number"><?php echo esc_html($stats->info); ?></div>
                <div class="stat-label">信息</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?php echo date('Y-m-d', strtotime($stats->first_log)); ?></div>
                <div class="stat-label">最早日志</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?php echo date('Y-m-d', strtotime($stats->last_log)); ?></div>
                <div class="stat-label">最新日志</div>
            </div>
        </div>
        <?php
    }
    
    /**
     * 显示筛选器
     */
    private function display_filters($current_filters) {
        ?>
        <div class="ncewpv-logs-filters">
            <form method="get">
                <input type="hidden" name="page" value="ncewpv-logs" />
                
                <div class="filter-group">
                    <label for="level-filter">级别:</label>
                    <select name="level" id="level-filter">
                        <option value="">所有级别</option>
                        <option value="DEBUG" <?php selected($current_filters['level'], 'DEBUG'); ?>>调试</option>
                        <option value="INFO" <?php selected($current_filters['level'], 'INFO'); ?>>信息</option>
                        <option value="WARNING" <?php selected($current_filters['level'], 'WARNING'); ?>>警告</option>
                        <option value="ERROR" <?php selected($current_filters['level'], 'ERROR'); ?>>错误</option>
                        <option value="CRITICAL" <?php selected($current_filters['level'], 'CRITICAL'); ?>>严重</option>
                    </select>
                </div>
                
                <div class="filter-group">
                    <label for="date-from">时间从:</label>
                    <input type="date" name="date_from" id="date-from" value="<?php echo esc_attr($current_filters['date_from']); ?>">
                </div>
                
                <div class="filter-group">
                    <label for="date-to">到:</label>
                    <input type="date" name="date_to" id="date-to" value="<?php echo esc_attr($current_filters['date_to']); ?>">
                </div>
                
                <div class="filter-group">
                    <label for="search">搜索:</label>
                    <input type="text" name="s" id="search" value="<?php echo esc_attr($current_filters['search']); ?>" placeholder="搜索消息或上下文...">
                </div>
                
                <div class="filter-group">
                    <button type="submit" class="button">筛选</button>
                    <a href="?page=ncewpv-logs" class="button button-secondary">重置</a>
                </div>
            </form>
        </div>
        <?php
    }
    
    /**
     * 显示工具栏
     */
    private function display_toolbar() {
        ?>
        <div class="ncewpv-logs-toolbar">
            <div class="toolbar-actions">
                <button type="button" id="ncewpv-clear-logs" class="button button-secondary">
                    清理旧日志
                </button>
                <select id="ncewpv-clear-days">
                    <option value="7">保留最近7天</option>
                    <option value="30" selected>保留最近30天</option>
                    <option value="90">保留最近90天</option>
                    <option value="365">保留最近1年</option>
                </select>
            </div>
            <div class="toolbar-info">
                <span class="dashicons dashicons-info"></span>
                日志文件位置: <code><?php echo WP_CONTENT_DIR; ?>/ncewpv-logs/</code>
            </div>
        </div>
        <?php
    }
    
    /**
     * 显示分页
     */
    private function display_pagination($current_page, $total_pages) {
        if ($total_pages <= 1) {
            return;
        }
        
        echo '<div class="tablenav bottom">';
        echo '<div class="tablenav-pages">';
        
        $base_url = remove_query_arg('paged');
        $page_links = array();
        
        // 上一页
        if ($current_page > 1) {
            $page_links[] = sprintf(
                '<a class="prev-page button" href="%s"><span class="screen-reader-text">上一页</span><span aria-hidden="true">‹</span></a>',
                esc_url(add_query_arg('paged', $current_page - 1, $base_url))
            );
        }
        
        // 页码
        for ($i = 1; $i <= $total_pages; $i++) {
            if ($i == $current_page) {
                $page_links[] = sprintf(
                    '<span class="current-page" aria-current="page">%d</span>',
                    $i
                );
            } else {
                $page_links[] = sprintf(
                    '<a class="page-numbers" href="%s">%d</a>',
                    esc_url(add_query_arg('paged', $i, $base_url)),
                    $i
                );
            }
        }
        
        // 下一页
        if ($current_page < $total_pages) {
            $page_links[] = sprintf(
                '<a class="next-page button" href="%s"><span class="screen-reader-text">下一页</span><span aria-hidden="true">›</span></a>',
                esc_url(add_query_arg('paged', $current_page + 1, $base_url))
            );
        }
        
        echo '<span class="pagination-links">' . implode('', $page_links) . '</span>';
        echo '</div>';
        echo '</div>';
    }
}

// 初始化日志管理
new NCEWPV_Admin_Logs();