<?php
/**
 * 订单管理页面
 */

// 防止直接访问
if (!defined('ABSPATH')) {
    exit;
}

class NCEWPV_Admin_Orders {
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_ajax_ncewpv_update_order_status', array($this, 'update_order_status'));
    }
    
    public function add_admin_menu() {
        add_submenu_page(
            'edit.php?post_type=page',
            '视频订单管理',
            '视频订单',
            'manage_options',
            'ncewpv-orders',
            array($this, 'orders_page')
        );
    }
    
    public function enqueue_scripts($hook) {
        if ($hook !== 'page_ncewpv-orders') {
            return;
        }
        
        wp_enqueue_style(
            'ncewpv-admin-orders',
            NCEWPV_PLUGIN_URL . 'assets/css/admin-orders.css',
            array(),
            NCEWPV_PLUGIN_VERSION
        );
        
        wp_enqueue_script(
            'ncewpv-admin-orders',
            NCEWPV_PLUGIN_URL . 'assets/js/admin-orders.js',
            array('jquery'),
            NCEWPV_PLUGIN_VERSION,
            true
        );
        
        wp_localize_script('ncewpv-admin-orders', 'ncewpv_orders', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('ncewpv_orders_nonce'),
            'confirm_status_change' => '确定要更改订单状态吗？'
        ));
    }
    
    public function orders_page() {
        $action = isset($_GET['action']) ? sanitize_text_field($_GET['action']) : '';
        $order_id = isset($_GET['order_id']) ? intval($_GET['order_id']) : 0;
        
        // 查看订单详情
        if ($action === 'view' && $order_id) {
            $this->view_order($order_id);
            return;
        }
        
        // 处理批量操作
        if (isset($_POST['bulk_action']) && $_POST['bulk_action'] !== '-1') {
            $this->handle_bulk_actions($_POST);
        }
        
        // 订单列表
        $this->display_orders_list();
    }
    
    private function display_orders_list() {
        global $wpdb;
        
        $per_page = 20;
        $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
        $offset = ($current_page - 1) * $per_page;
        
        // 筛选参数
        $status = isset($_GET['status']) ? sanitize_text_field($_GET['status']) : '';
        $user_id = isset($_GET['user_id']) ? intval($_GET['user_id']) : 0;
        $post_id = isset($_GET['post_id']) ? intval($_GET['post_id']) : 0;
        $date_from = isset($_GET['date_from']) ? sanitize_text_field($_GET['date_from']) : '';
        $date_to = isset($_GET['date_to']) ? sanitize_text_field($_GET['date_to']) : '';
        $search = isset($_GET['s']) ? sanitize_text_field($_GET['s']) : '';
        
        // 构建查询条件
        $where = array('1=1');
        
        if (!empty($status)) {
            $where[] = $wpdb->prepare("status = %s", $status);
        }
        
        if ($user_id > 0) {
            $where[] = $wpdb->prepare("user_id = %d", $user_id);
        }
        
        if ($post_id > 0) {
            $where[] = $wpdb->prepare("post_id = %d", $post_id);
        }
        
        if (!empty($date_from)) {
            $where[] = $wpdb->prepare("created_at >= %s", $date_from . ' 00:00:00');
        }
        
        if (!empty($date_to)) {
            $where[] = $wpdb->prepare("created_at <= %s", $date_to . ' 23:59:59');
        }
        
        if (!empty($search)) {
            $search_term = '%' . $wpdb->esc_like($search) . '%';
            $where[] = $wpdb->prepare(
                "(paypal_order_id LIKE %s OR id LIKE %s)",
                $search_term,
                $search_term
            );
        }
        
        $where_sql = implode(' AND ', $where);
        
        // 获取订单总数
        $total_orders = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->prefix}ncewpv_orders WHERE {$where_sql}"
        );
        
        // 获取订单列表
        $orders = $wpdb->get_results(
            "SELECT * FROM {$wpdb->prefix}ncewpv_orders 
            WHERE {$where_sql} 
            ORDER BY created_at DESC 
            LIMIT {$per_page} OFFSET {$offset}"
        );
        
        $total_pages = ceil($total_orders / $per_page);
        
        ?>
        <div class="wrap ncewpv-orders-wrap">
            <h1 class="wp-heading-inline">视频订单管理</h1>
            <a href="<?php echo admin_url('edit.php?post_type=page&page=ncewpv-orders'); ?>" class="page-title-action">所有订单</a>
            
            <?php $this->display_orders_filters($status, $user_id, $post_id, $date_from, $date_to); ?>
            
            <form method="post">
                <div class="tablenav top">
                    <div class="bulk-actions alignleft">
                        <select name="bulk_action">
                            <option value="-1">批量操作</option>
                            <option value="mark_completed">标记为已完成</option>
                            <option value="mark_refunded">标记为已退款</option>
                            <option value="mark_cancelled">标记为已取消</option>
                            <option value="delete">删除</option>
                        </select>
                        <input type="submit" class="button action" value="应用">
                    </div>
                    <?php
                    echo paginate_links(array(
                        'base' => add_query_arg('paged', '%#%'),
                        'format' => '',
                        'prev_text' => __('&laquo;'),
                        'next_text' => __('&raquo;'),
                        'total' => $total_pages,
                        'current' => $current_page
                    ));
                    ?>
                    <br class="clear">
                </div>
                
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th width="50"><input type="checkbox" id="select-all"></th>
                            <th>订单ID</th>
                            <th>用户</th>
                            <th>视频</th>
                            <th>金额</th>
                            <th>PayPal订单ID</th>
                            <th>状态</th>
                            <th>创建时间</th>
                            <th>操作</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($orders)): ?>
                            <tr>
                                <td colspan="9" class="text-center">没有找到订单记录</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($orders as $order): ?>
                                <tr>
                                    <td><input type="checkbox" name="order_ids[]" value="<?php echo $order->id; ?>"></td>
                                    <td><?php echo $order->id; ?></td>
                                    <td>
                                        <?php 
                                        $user = get_userdata($order->user_id);
                                        if ($user) {
                                            echo $user->display_name . ' (ID: ' . $order->user_id . ')';
                                        } else {
                                            echo '用户已删除 (ID: ' . $order->user_id . ')';
                                        }
                                        ?>
                                    </td>
                                    <td>
                                        <?php 
                                        $post = get_post($order->post_id);
                                        if ($post) {
                                            echo '<a href="' . get_edit_post_link($post->ID) . '" target="_blank">' . $post->post_title . '</a>';
                                        } else {
                                            echo '文章已删除 (ID: ' . $order->post_id . ')';
                                        }
                                        ?>
                                    </td>
                                    <td><?php echo number_format($order->order_amount, 2); ?> 元</td>
                                    <td><?php echo $order->paypal_order_id ? esc_html($order->paypal_order_id) : '-'; ?></td>
                                    <td>
                                        <span class="ncewpv-order-status status-<?php echo esc_attr($order->status); ?>">
                                            <?php echo $this->get_status_label($order->status); ?>
                                        </span>
                                    </td>
                                    <td><?php echo date('Y-m-d H:i', strtotime($order->created_at)); ?></td>
                                    <td>
                                        <a href="<?php echo admin_url('edit.php?post_type=page&page=ncewpv-orders&action=view&order_id=' . $order->id); ?>" class="button button-small">查看</a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
                
                <div class="tablenav bottom">
                    <?php
                    echo paginate_links(array(
                        'base' => add_query_arg('paged', '%#%'),
                        'format' => '',
                        'prev_text' => __('&laquo;'),
                        'next_text' => __('&raquo;'),
                        'total' => $total_pages,
                        'current' => $current_page
                    ));
                    ?>
                    <br class="clear">
                </div>
            </form>
        </div>
        <?php
    }
    
    private function view_order($order_id) {
        global $wpdb;
        
        $order = $wpdb->get_row(
            $wpdb->prepare("SELECT * FROM {$wpdb->prefix}ncewpv_orders WHERE id = %d", $order_id)
        );
        
        if (!$order) {
            wp_die('订单不存在或已被删除');
        }
        
        // 获取相关日志
        $error_handler = NCEWPV_Error_Handler::get_instance();
        $logs = $error_handler->get_logs(array(
            'search' => $order->paypal_order_id,
            'limit' => 100
        ));
        
        ?>
        <div class="wrap ncewpv-order-detail">
            <h1 class="wp-heading-inline">订单详情 #<?php echo $order->id; ?></h1>
            <a href="<?php echo admin_url('edit.php?post_type=page&page=ncewpv-orders'); ?>" class="page-title-action">返回列表</a>
            
            <div class="order-meta-boxes">
                <div class="order-meta-box">
                    <h3>订单信息</h3>
                    <table class="form-table">
                        <tr>
                            <th width="30%">订单ID</th>
                            <td><?php echo $order->id; ?></td>
                        </tr>
                        <tr>
                            <th>创建时间</th>
                            <td><?php echo $order->created_at; ?></td>
                        </tr>
                        <tr>
                            <th>更新时间</th>
                            <td><?php echo $order->updated_at; ?></td>
                        </tr>
                        <tr>
                            <th>状态</th>
                            <td>
                                <span class="ncewpv-order-status status-<?php echo esc_attr($order->status); ?>">
                                    <?php echo $this->get_status_label($order->status); ?>
                                </span>
                                <div class="status-changer">
                                    <select name="new_status" class="order-status-select" data-order-id="<?php echo $order->id; ?>">
                                        <option value="pending" <?php selected($order->status, 'pending'); ?>>待支付</option>
                                        <option value="completed" <?php selected($order->status, 'completed'); ?>>已完成</option>
                                        <option value="refunded" <?php selected($order->status, 'refunded'); ?>>已退款</option>
                                        <option value="cancelled" <?php selected($order->status, 'cancelled'); ?>>已取消</option>
                                    </select>
                                    <button class="button button-small update-status">更新状态</button>
                                </div>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <div class="order-meta-box">
                    <h3>用户信息</h3>
                    <table class="form-table">
                        <tr>
                            <th width="30%">用户ID</th>
                            <td>
                                <?php 
                                $user = get_userdata($order->user_id);
                                if ($user) {
                                    echo $order->user_id . ' - <a href="' . get_edit_user_link($order->user_id) . '" target="_blank">' . $user->display_name . '</a>';
                                    echo '<br>' . $user->user_email;
                                } else {
                                    echo '用户已删除 (ID: ' . $order->user_id . ')';
                                }
                                ?>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <div class="order-meta-box">
                    <h3>视频信息</h3>
                    <table class="form-table">
                        <tr>
                            <th width="30%">视频文章</th>
                            <td>
                                <?php 
                                $post = get_post($order->post_id);
                                if ($post) {
                                    echo $order->post_id . ' - <a href="' . get_edit_post_link($post->ID) . '" target="_blank">' . $post->post_title . '</a>';
                                    echo '<br><a href="' . get_permalink($post->ID) . '" target="_blank">查看前台</a>';
                                } else {
                                    echo '文章已删除 (ID: ' . $order->post_id . ')';
                                }
                                ?>
                            </td>
                        </tr>
                        <tr>
                            <th>订单金额</th>
                            <td><?php echo number_format($order->order_amount, 2); ?> 元</td>
                        </tr>
                    </table>
                </div>
                
                <div class="order-meta-box">
                    <h3>支付信息</h3>
                    <table class="form-table">
                        <tr>
                            <th width="30%">PayPal订单ID</th>
                            <td><?php echo $order->paypal_order_id ? esc_html($order->paypal_order_id) : '未设置'; ?></td>
                        </tr>
                        <tr>
                            <th>PayPal详情</th>
                            <td>
                                <?php if ($order->paypal_order_id): ?>
                                    <a href="https://www.paypal.com/activity/payment/<?php echo $order->paypal_order_id; ?>" target="_blank">查看PayPal交易</a>
                                <?php else: ?>
                                    无PayPal交易信息
                                <?php endif; ?>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <div class="ncewpv-debug-info">
                <h3>调试日志（最近20条）</h3>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th>时间</th>
                            <th>操作</th>
                            <th>用户</th>
                            <th>状态</th>
                            <th>详情</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach (array_slice($logs, -20) as $log): ?>
                            <tr>
                                <td><?php echo date('Y-m-d H:i:s', strtotime($log->created_at)); ?></td>
                                <td><?php echo esc_html($log->message); ?></td>
                                <td>
                                    <?php 
                                    $user_id = $log->user_id;
                                    if ($user_id) {
                                        $user = get_userdata($user_id);
                                        echo $user ? esc_html($user->display_name) : '用户已删除';
                                    } else {
                                        echo '系统';
                                    }
                                    ?>
                                </td>
                                <td>
                                    <?php 
                                    $status = strtolower($log->level);
                                    $status_class = $status === 'error' || $status === 'critical' ? 'status-error' : 'status-success';
                                    echo '<span class="ncewpv-log-status ' . $status_class . '">' . esc_html($log->level) . '</span>';
                                    ?>
                                </td>
                                <td>
                                    <?php if ($log->context): ?>
                                        <button type="button" class="button button-small view-context" 
                                                data-context="<?php echo esc_attr($log->context); ?>">
                                            详情
                                        </button>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
            <!-- 上下文详情模态框 -->
            <div id="ncewpv-log-context-modal" class="ncewpv-modal" style="display: none;">
                <div class="ncewpv-modal-overlay"></div>
                <div class="ncewpv-modal-content">
                    <div class="ncewpv-modal-header">
                        <h3>日志详情</h3>
                        <button class="ncewpv-modal-close">&times;</button>
                    </div>
                    <div class="ncewpv-modal-body">
                        <pre id="ncewpv-log-context-content"></pre>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    private function display_orders_filters($current_status, $current_user, $current_post, $current_date_from, $current_date_to) {
        ?>
        <div class="ncewpv-filters">
            <form method="get">
                <input type="hidden" name="post_type" value="page">
                <input type="hidden" name="page" value="ncewpv-orders">
                
                <div class="filter-group">
                    <label for="order-status">状态:</label>
                    <select name="status" id="order-status">
                        <option value="">所有状态</option>
                        <option value="pending" <?php selected($current_status, 'pending'); ?>>待支付</option>
                        <option value="completed" <?php selected($current_status, 'completed'); ?>>已完成</option>
                        <option value="refunded" <?php selected($current_status, 'refunded'); ?>>已退款</option>
                        <option value="cancelled" <?php selected($current_status, 'cancelled'); ?>>已取消</option>
                    </select>
                </div>
                
                <div class="filter-group">
                    <label for="user-id">用户ID:</label>
                    <input type="number" name="user_id" id="user-id" value="<?php echo $current_user; ?>" placeholder="用户ID">
                </div>
                
                <div class="filter-group">
                    <label for="post-id">视频ID:</label>
                    <input type="number" name="post_id" id="post-id" value="<?php echo $current_post; ?>" placeholder="文章ID">
                </div>
                
                <div class="filter-group">
                    <label for="date-from">从:</label>
                    <input type="date" name="date_from" id="date-from" value="<?php echo $current_date_from; ?>">
                </div>
                
                <div class="filter-group">
                    <label for="date-to">到:</label>
                    <input type="date" name="date_to" id="date-to" value="<?php echo $current_date_to; ?>">
                </div>
                
                <div class="filter-group">
                    <label>&nbsp;</label>
                    <button type="submit" class="button">筛选</button>
                    <a href="<?php echo admin_url('edit.php?post_type=page&page=ncewpv-orders'); ?>" class="button button-secondary">重置</a>
                </div>
            </form>
        </div>
        <?php
    }
    
    private function handle_bulk_actions($data) {
        if (!isset($data['order_ids']) || !is_array($data['order_ids'])) {
            return;
        }
        
        $action = sanitize_text_field($data['bulk_action']);
        $order_ids = array_map('intval', $data['order_ids']);
        
        global $wpdb;
        
        foreach ($order_ids as $order_id) {
            switch ($action) {
                case 'mark_completed':
                    $wpdb->update(
                        $wpdb->prefix . 'ncewpv_orders',
                        array('status' => 'completed'),
                        array('id' => $order_id)
                    );
                    break;
                    
                case 'mark_refunded':
                    $wpdb->update(
                        $wpdb->prefix . 'ncewpv_orders',
                        array('status' => 'refunded'),
                        array('id' => $order_id)
                    );
                    break;
                    
                case 'mark_cancelled':
                    $wpdb->update(
                        $wpdb->prefix . 'ncewpv_orders',
                        array('status' => 'cancelled'),
                        array('id' => $order_id)
                    );
                    break;
                    
                case 'delete':
                    $wpdb->delete(
                        $wpdb->prefix . 'ncewpv_orders',
                        array('id' => $order_id)
                    );
                    break;
            }
        }
    }
    
    public function update_order_status() {
        // 验证nonce
        if (!wp_verify_nonce($_POST['nonce'], 'ncewpv_orders_nonce')) {
            wp_send_json_error('安全验证失败');
        }
        
        $order_id = isset($_POST['order_id']) ? intval($_POST['order_id']) : 0;
        $new_status = isset($_POST['new_status']) ? sanitize_text_field($_POST['new_status']) : '';
        
        if (!$order_id || !in_array($new_status, array('pending', 'completed', 'refunded', 'cancelled'))) {
            wp_send_json_error('参数错误');
        }
        
        global $wpdb;
        $result = $wpdb->update(
            $wpdb->prefix . 'ncewpv_orders',
            array('status' => $new_status),
            array('id' => $order_id)
        );
        
        if ($result !== false) {
            wp_send_json_success(array(
                'message' => '状态已更新',
                'status_label' => $this->get_status_label($new_status)
            ));
        } else {
            wp_send_json_error('更新失败');
        }
    }
    
    private function get_status_label($status) {
        $labels = array(
            'pending' => '待支付',
            'completed' => '已完成',
            'refunded' => '已退款',
            'cancelled' => '已取消'
        );
        
        return isset($labels[$status]) ? $labels[$status] : $status;
    }
}

// 初始化订单管理页面
new NCEWPV_Admin_Orders();