<?php
/**
 * 管理员设置页面
 */

// 防止直接访问
if (!defined('ABSPATH')) {
    exit;
}

class NCEWPV_Admin_Settings {

    private $settings_page = 'ncewpv-settings';
    private $options;

    public function __construct() {
        add_action('admin_menu', array($this, 'add_plugin_page'));
        add_action('admin_init', array($this, 'page_init'));
        add_action('admin_footer', array($this, 'add_test_connection_script')); // 添加JS脚本
    }

    public function add_plugin_page() {
        add_options_page(
            '收费视频设置',
            '收费视频',
            'manage_options',
            $this->settings_page,
            array($this, 'create_admin_page')
        );
    }

    public function create_admin_page() {
        $this->options = get_option('ncewpv_settings');
        ?>
        <div class="wrap ncewpv-settings-wrap">
            <h1>收费视频设置</h1>
            <p>配置PayPal支付参数和收费设置</p>
            
            <form method="post" action="options.php">
                <?php
                settings_fields('ncewpv_settings_group');
                do_settings_sections($this->settings_page);
                submit_button();
                ?>
            </form>
            
            <div class="ncewpv-settings-info">
                <h3>使用说明</h3>
                <ul>
                    <li><strong>运行环境：</strong>开发测试时选择"沙盒环境"，正式上线时选择"生产环境"</li>
                    <li><strong>Client ID & Secret：</strong>从PayPal开发者平台获取</li>
                    <li><strong>费率系数：</strong>用于计算最终价格，例如：1.03 表示包含3%的平台手续费</li>
                    <li><strong>固定费：</strong>每笔订单的固定手续费（元）</li>
                </ul>
                
                <h3>PayPal API凭证获取</h3>
                <ol>
                    <li>访问 <a href="https://developer.paypal.com" target="_blank">PayPal开发者平台</a></li>
                    <li>登录您的PayPal账户</li>
                    <li>进入"Dashboard" → "REST API apps"</li>
                    <li>创建应用或使用现有应用</li>
                    <li>复制"Client ID"和"Secret"到此页面</li>
                </ol>
            </div>
        </div>
        <?php
    }

    public function page_init() {
        register_setting(
            'ncewpv_settings_group',
            'ncewpv_settings',
            array($this, 'sanitize')
        );
        
        // PayPal配置部分
        add_settings_section(
            'ncewpv_paypal_section',
            'PayPal配置',
            array($this, 'paypal_section_info'),
            $this->settings_page
        );
        
        add_settings_field(
            'environment',
            '运行环境',
            array($this, 'environment_callback'),
            $this->settings_page,
            'ncewpv_paypal_section'
        );
        
        add_settings_field(
            'client_id',
            'Client ID',
            array($this, 'client_id_callback'),
            $this->settings_page,
            'ncewpv_paypal_section'
        );
        
        add_settings_field(
            'client_secret',
            'Secret',
            array($this, 'client_secret_callback'),
            $this->settings_page,
            'ncewpv_paypal_section'
        );
        
        // 费用配置部分
        add_settings_section(
            'ncewpv_fee_section',
            '费用设置',
            array($this, 'fee_section_info'),
            $this->settings_page
        );
        
        add_settings_field(
            'fee_rate',
            '费率系数',
            array($this, 'fee_rate_callback'),
            $this->settings_page,
            'ncewpv_fee_section'
        );
        
        add_settings_field(
            'fixed_fee',
            '固定费（元）',
            array($this, 'fixed_fee_callback'),
            $this->settings_page,
            'ncewpv_fee_section'
        );
        
        // 测试部分
        add_settings_section(
            'ncewpv_test_section',
            '测试连接',
            array($this, 'test_section_info'),
            $this->settings_page
        );
        
        add_settings_field(
            'test_connection',
            'PayPal连接测试',
            array($this, 'test_connection_callback'),
            $this->settings_page,
            'ncewpv_test_section'
        );
    }

    public function sanitize($input) {
        $sanitized_input = array();
        
        if (isset($input['environment'])) {
            $sanitized_input['environment'] = sanitize_text_field($input['environment']);
        }
        
        if (isset($input['client_id'])) {
            $sanitized_input['client_id'] = sanitize_text_field($input['client_id']);
        }
        
        if (isset($input['client_secret'])) {
            $sanitized_input['client_secret'] = sanitize_text_field($input['client_secret']);
        }
        
        if (isset($input['fee_rate'])) {
            $fee_rate = floatval($input['fee_rate']);
            $sanitized_input['fee_rate'] = ($fee_rate >= 1) ? $fee_rate : 1.0;
        }
        
        if (isset($input['fixed_fee'])) {
            $fixed_fee = floatval($input['fixed_fee']);
            $sanitized_input['fixed_fee'] = ($fixed_fee >= 0) ? $fixed_fee : 0;
        }
        
        // 如果配置发生变化，清除PayPal token缓存
        $old_options = get_option('ncewpv_settings', array());
        if (($old_options['client_id'] ?? '') !== ($sanitized_input['client_id'] ?? '') || 
            ($old_options['client_secret'] ?? '') !== ($sanitized_input['client_secret'] ?? '') ||
            ($old_options['environment'] ?? '') !== ($sanitized_input['environment'] ?? '')) {
            delete_transient('ncewpv_paypal_access_token');
        }
        
        return $sanitized_input;
    }

    public function paypal_section_info() {
        echo '配置PayPal API凭证和运行环境';
    }

    public function fee_section_info() {
        echo '设置平台手续费计算规则';
    }

    public function test_section_info() {
        echo '测试PayPal API连接状态';
    }

    public function environment_callback() {
        $environment = isset($this->options['environment']) ? $this->options['environment'] : 'sandbox';
        ?>
        <select name="ncewpv_settings[environment]" id="environment">
            <option value="sandbox" <?php selected($environment, 'sandbox'); ?>>沙盒环境 (测试)</option>
            <option value="live" <?php selected($environment, 'live'); ?>>生产环境</option>
        </select>
        <p class="description">开发测试阶段请选择沙盒环境</p>
        <?php
    }

    public function client_id_callback() {
        printf(
            '<input type="text" id="client_id" name="ncewpv_settings[client_id]" value="%s" class="regular-text" />',
            isset($this->options['client_id']) ? esc_attr($this->options['client_id']) : ''
        );
    }

    public function client_secret_callback() {
        printf(
            '<input type="password" id="client_secret" name="ncewpv_settings[client_secret]" value="%s" class="regular-text" />',
            isset($this->options['client_secret']) ? esc_attr($this->options['client_secret']) : ''
        );
    }

    public function fee_rate_callback() {
        $fee_rate = isset($this->options['fee_rate']) ? $this->options['fee_rate'] : '1.03';
        ?>
        <input type="number" id="fee_rate" name="ncewpv_settings[fee_rate]" 
               value="<?php echo esc_attr($fee_rate); ?>" step="0.01" min="1.00" max="2.00" />
        <p class="description">
            最终价格 = (视频价格 × 费率系数) + 固定费<br>
            例如：费率系数 1.03 表示包含3%手续费
        </p>
        <?php
    }

    public function fixed_fee_callback() {
        $fixed_fee = isset($this->options['fixed_fee']) ? $this->options['fixed_fee'] : '0';
        ?>
        <input type="number" id="fixed_fee" name="ncewpv_settings[fixed_fee]" 
               value="<?php echo esc_attr($fixed_fee); ?>" step="0.01" min="0" max="10" />
        <p class="description">每笔订单的固定手续费，单位：元</p>
        <?php
    }

    public function test_connection_callback() {
        ?>
        <button type="button" id="ncewpv-test-connection" class="button button-secondary">测试PayPal连接</button>
        <span id="ncewpv-test-result" style="margin-left: 10px;"></span>
        <p class="description">点击测试当前PayPal配置是否有效</p>
        <?php
    }

    /**
     * 获取PayPal基础URL
     */
    public static function get_paypal_base_url() {
        $options = get_option('ncewpv_settings');
        $environment = isset($options['environment']) ? $options['environment'] : 'sandbox';
        
        return $environment === 'live' 
            ? 'https://api.paypal.com' 
            : 'https://api.sandbox.paypal.com';
    }

    /**
     * 获取PayPal JS SDK URL
     */
    public static function get_paypal_js_sdk_url() {
        $options = get_option('ncewpv_settings');
        $environment = isset($options['environment']) ? $options['environment'] : 'sandbox';
        $client_id = isset($options['client_id']) ? $options['client_id'] : '';
        
        $base_url = $environment === 'live'
            ? 'https://www.paypal.com/sdk/js'
            : 'https://www.sandbox.paypal.com/sdk/js';
            
        return $base_url . '?client-id=' . $client_id . '&currency=USD&intent=capture';
    }

    /**
     * 计算最终价格（包含手续费）
     */
    public static function calculate_final_price($original_price) {
        $options = get_option('ncewpv_settings');
        $fee_rate = isset($options['fee_rate']) ? floatval($options['fee_rate']) : 1.03;
        $fixed_fee = isset($options['fixed_fee']) ? floatval($options['fixed_fee']) : 0;
        
        $final_price = ($original_price * $fee_rate) + $fixed_fee;
        
        // 四舍五入到2位小数
        return round($final_price, 2);
    }

    /**
     * 添加测试连接的JavaScript
     */
    public function add_test_connection_script() {
        // 只在设置页面加载脚本
        if (!isset($_GET['page']) || $_GET['page'] !== $this->settings_page) {
            return;
        }
        ?>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            $('#ncewpv-test-connection').on('click', function() {
                var $button = $(this);
                var $result = $('#ncewpv-test-result');
                
                // 显示加载状态
                $button.prop('disabled', true).text('测试中...');
                $result.html('<span class="spinner is-active" style="float:none;"></span>');
                
                // 发送AJAX请求
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'ncewpv_test_paypal_connection',
                        nonce: '<?php echo wp_create_nonce('ncewpv_admin_nonce'); ?>'
                    },
                    dataType: 'json',
                    success: function(response) {
                        if (response.success) {
                            $result.html('<span style="color:green;">✓ 连接成功</span>');
                        } else {
                            $result.html('<span style="color:red;">✗ 失败: ' + (response.data || '未知错误') + '</span>');
                        }
                        $button.prop('disabled', false).text('测试PayPal连接');
                    },
                    error: function(xhr, status, error) {
                        $result.html('<span style="color:red;">✗ 请求失败: ' + error + '</span>');
                        $button.prop('disabled', false).text('测试PayPal连接');
                    },
                    timeout: 15000 // 15秒超时
                });
            });
        });
        </script>
        <?php
    }
}

// 初始化设置页面
new NCEWPV_Admin_Settings();