<?php
/**
 * AJAX请求处理
 */

// 防止直接访问
if (!defined('ABSPATH')) {
    exit;
}

class NCEWPV_Ajax_Handler {
    
    public function __construct() {
        // PayPal订单相关
        add_action('wp_ajax_ncewpv_create_paypal_order', array($this, 'create_paypal_order'));
        add_action('wp_ajax_nopriv_ncewpv_create_paypal_order', array($this, 'create_paypal_order'));
        
        add_action('wp_ajax_ncewpv_capture_paypal_order', array($this, 'capture_paypal_order'));
        add_action('wp_ajax_nopriv_ncewpv_capture_paypal_order', array($this, 'capture_paypal_order'));
        
        // 购买状态检查
        add_action('wp_ajax_ncewpv_check_purchase_status', array($this, 'check_purchase_status'));
        add_action('wp_ajax_nopriv_ncewpv_check_purchase_status', array($this, 'check_purchase_status'));
        
        // 管理员测试连接
        add_action('wp_ajax_ncewpv_test_paypal_connection', array($this, 'test_paypal_connection'));
        
        // WooCommerce产品创建
        add_action('wp_ajax_ncewpv_create_woocommerce_product', array($this, 'create_woocommerce_product'));
    }
    
    /**
     * 创建PayPal订单
     */
    public function create_paypal_order() {
        // 验证nonce
        if (!wp_verify_nonce($_POST['nonce'], 'ncewpv_ajax_nonce')) {
            wp_send_json_error('安全验证失败');
        }
        
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        $amount = isset($_POST['amount']) ? floatval($_POST['amount']) : 0;
        
        if (!$post_id || $amount <= 0) {
            if (function_exists('ncewpv_log_error')) {
                ncewpv_log_error('创建PayPal订单参数错误', [
                    'post_id' => $post_id,
                    'amount' => $amount,
                    'user_id' => get_current_user_id()
                ]);
            }
            wp_send_json_error('参数错误：文章ID或金额无效');
        }
        
        // 检查用户是否已登录
        if (!is_user_logged_in()) {
            wp_send_json_error('请先登录后再进行购买');
        }
        
        $user_id = get_current_user_id();
        
        // 检查用户是否已购买
        if ($this->user_has_purchased($post_id, $user_id)) {
            if (function_exists('ncewpv_log_warning')) {
                ncewpv_log_warning('用户尝试重复购买', [
                    'post_id' => $post_id,
                    'user_id' => $user_id
                ]);
            }
            wp_send_json_error('您已购买过此视频，无需重复购买');
        }
        
        // 检查文章是否存在且包含收费视频块
        $post = get_post($post_id);
        if (!$post || !has_block('ncewpv/paid-video', $post)) {
            if (function_exists('ncewpv_log_error')) {
                ncewpv_log_error('文章不存在或不包含收费视频', [
                    'post_id' => $post_id,
                    'user_id' => $user_id
                ]);
            }
            wp_send_json_error('文章不存在或不包含收费视频');
        }
        
        try {
            require_once NCEWPV_PLUGIN_PATH . 'includes/paypal-handler.php';
            $paypal_handler = new NCEWPV_PayPal_Handler();
            
            // 使用管理员设置计算最终价格
            require_once NCEWPV_PLUGIN_PATH . 'includes/admin-settings.php';
            $final_amount = NCEWPV_Admin_Settings::calculate_final_price($amount);
            
            if (function_exists('ncewpv_log_info')) {
                ncewpv_log_info('开始创建PayPal订单', [
                    'user_id' => $user_id,
                    'post_id' => $post_id,
                    'original_amount' => $amount,
                    'final_amount' => $final_amount
                ]);
            }
            
            // 创建PayPal订单
            $order = $paypal_handler->create_order($final_amount, $post_id);
            
            if (function_exists('ncewpv_log_info')) {
                ncewpv_log_info('PayPal订单创建成功', [
                    'user_id' => $user_id,
                    'post_id' => $post_id,
                    'paypal_order_id' => $order['id'],
                    'amount' => $final_amount
                ]);
            }
            
            wp_send_json_success($order);
            
        } catch (Exception $e) {
            if (function_exists('ncewpv_log_error')) {
                ncewpv_log_error('创建PayPal订单失败', [
                    'user_id' => $user_id,
                    'post_id' => $post_id,
                    'amount' => $amount,
                    'error' => $e->getMessage()
                ]);
            }
            
            wp_send_json_error($e->getMessage());
        }
    }
    
    /**
     * 捕获PayPal订单
     */
    public function capture_paypal_order() {
        // 验证nonce
        if (!wp_verify_nonce($_POST['nonce'], 'ncewpv_ajax_nonce')) {
            wp_send_json_error('安全验证失败');
        }
        
        $order_id = isset($_POST['order_id']) ? sanitize_text_field($_POST['order_id']) : '';
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        
        if (empty($order_id) || !$post_id) {
            if (function_exists('ncewpv_log_error')) {
                ncewpv_log_error('捕获PayPal订单参数错误', [
                    'order_id' => $order_id,
                    'post_id' => $post_id,
                    'user_id' => get_current_user_id()
                ]);
            }
            wp_send_json_error('参数错误：订单ID或文章ID无效');
        }
        
        // 检查用户是否已登录
        if (!is_user_logged_in()) {
            wp_send_json_error('请先登录');
        }
        
        $user_id = get_current_user_id();
        
        try {
            require_once NCEWPV_PLUGIN_PATH . 'includes/paypal-handler.php';
            $paypal_handler = new NCEWPV_PayPal_Handler();
            
            if (function_exists('ncewpv_log_info')) {
                ncewpv_log_info('开始捕获PayPal订单', [
                    'user_id' => $user_id,
                    'post_id' => $post_id,
                    'paypal_order_id' => $order_id
                ]);
            }
            
            // 捕获PayPal订单
            $capture_result = $paypal_handler->capture_order($order_id, $post_id);
            
            if (function_exists('ncewpv_log_info')) {
                ncewpv_log_info('PayPal订单捕获成功', [
                    'user_id' => $user_id,
                    'post_id' => $post_id,
                    'paypal_order_id' => $order_id,
                    'capture_id' => $capture_result['id'],
                    'amount' => $capture_result['purchase_units'][0]['amount']['value']
                ]);
            }
            
            // 发送购买成功邮件
            $this->send_purchase_success_email($user_id, $post_id, $capture_result);
            
            wp_send_json_success(array(
                'message' => '支付成功',
                'capture_data' => $capture_result
            ));
            
        } catch (Exception $e) {
            if (function_exists('ncewpv_log_error')) {
                ncewpv_log_error('捕获PayPal订单失败', [
                    'user_id' => $user_id,
                    'post_id' => $post_id,
                    'paypal_order_id' => $order_id,
                    'error' => $e->getMessage()
                ]);
            }
            
            wp_send_json_error($e->getMessage());
        }
    }
    
    /**
     * 检查购买状态
     */
    public function check_purchase_status() {
        // 验证nonce
        if (!wp_verify_nonce($_POST['nonce'], 'ncewpv_ajax_nonce')) {
            wp_send_json_error('安全验证失败');
        }
        
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        $user_id = get_current_user_id();
        
        if (!$post_id) {
            if (function_exists('ncewpv_log_error')) {
                ncewpv_log_error('检查购买状态参数错误', ['post_id' => $post_id]);
            }
            wp_send_json_error('参数错误：文章ID无效');
        }
        
        if (!$user_id) {
            wp_send_json_success(array(
                'has_purchased' => false,
                'message' => '用户未登录'
            ));
        }
        
        $has_purchased = $this->user_has_purchased($post_id, $user_id);
        
        if (function_exists('ncewpv_log_debug')) {
            ncewpv_log_debug('检查用户购买状态', [
                'user_id' => $user_id,
                'post_id' => $post_id,
                'has_purchased' => $has_purchased
            ]);
        }
        
        wp_send_json_success(array(
            'has_purchased' => $has_purchased,
            'user_id' => $user_id,
            'post_id' => $post_id
        ));
    }
    
    /**
     * 测试PayPal连接
     */
    public function test_paypal_connection() {
        // 检查权限
        if (!current_user_can('manage_options')) {
            wp_send_json_error('权限不足');
        }
        
        // 验证nonce
        if (!wp_verify_nonce($_POST['nonce'], 'ncewpv_admin_nonce')) {
            wp_send_json_error('安全验证失败');
        }
        
        try {
            require_once NCEWPV_PLUGIN_PATH . 'includes/paypal-handler.php';
            $paypal_handler = new NCEWPV_PayPal_Handler();
            
            // 验证必要配置
            $settings = get_option('ncewpv_settings');
            if (empty($settings['client_id']) || empty($settings['client_secret'])) {
                throw new Exception('请先填写Client ID和Secret');
            }
            
            if (function_exists('ncewpv_log_info')) {
                ncewpv_log_info('开始测试PayPal连接', ['user_id' => get_current_user_id()]);
            }
            
            $result = $paypal_handler->test_connection();
            
            if ($result['success']) {
                if (function_exists('ncewpv_log_info')) {
                    ncewpv_log_info('PayPal连接测试成功', ['user_id' => get_current_user_id()]);
                }
                wp_send_json_success('PayPal连接测试成功');
            } else {
                if (function_exists('ncewpv_log_error')) {
                    ncewpv_log_error('PayPal连接测试失败', [
                        'user_id' => get_current_user_id(),
                        'error' => $result['message']
                    ]);
                }
                wp_send_json_error($result['message']);
            }
            
        } catch (Exception $e) {
            if (function_exists('ncewpv_log_error')) {
                ncewpv_log_error('PayPal连接测试异常', [
                    'user_id' => get_current_user_id(),
                    'error' => $e->getMessage()
                ]);
            }
            
            wp_send_json_error($e->getMessage());
        }
    }
    
    /**
     * 创建WooCommerce产品
     */
    public function create_woocommerce_product() {
        // 实现WooCommerce产品创建逻辑
    }

    /**
     * 检查用户是否已购买
     */
    private function user_has_purchased($post_id, $user_id) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'ncewpv_orders';
        
        $count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table_name WHERE post_id = %d AND user_id = %d AND status = 'completed'",
            $post_id, $user_id
        ));
        
        return $count > 0;
    }

    /**
     * 发送购买成功邮件
     */
    private function send_purchase_success_email($user_id, $post_id, $capture_result) {
        // 实现邮件发送逻辑
        $user = get_userdata($user_id);
        $post = get_post($post_id);
        
        if ($user && $post) {
            $to = $user->user_email;
            $subject = '视频购买成功 - ' . get_bloginfo('name');
            $message = "
                <html>
                <head>
                    <style>
                        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                        .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                        .header { background: #28a745; color: white; padding: 20px; text-align: center; border-radius: 8px 8px 0 0; }
                        .content { padding: 20px; background: #f9f9f9; }
                        .footer { padding: 20px; text-align: center; font-size: 12px; color: #666; }
                    </style>
                </head>
                <body>
                    <div class='container'>
                        <div class='header'>
                            <h1>购买成功！</h1>
                        </div>
                        <div class='content'>
                            <p>亲爱的 " . $user->display_name . "，</p>
                            <p>您已成功购买视频：<strong>" . $post->post_title . "</strong></p>
                            <p>支付金额：<strong>" . $capture_result['purchase_units'][0]['amount']['value'] . " USD</strong></p>
                            <p>订单号：" . $capture_result['id'] . "</p>
                            <p>您现在可以随时观看完整视频内容。</p>
                            <p><a href='" . get_permalink($post_id) . "' style='background: #007cba; color: white; padding: 10px 20px; text-decoration: none; border-radius: 4px; display: inline-block;'>立即观看视频</a></p>
                        </div>
                        <div class='footer'>
                            此邮件由 " . get_bloginfo('name') . " 自动发送
                        </div>
                    </div>
                </body>
                </html>
            ";
            
            $headers = array(
                'Content-Type: text/html; charset=UTF-8',
                'From: ' . get_bloginfo('name') . ' <' . get_option('admin_email') . '>'
            );
            
            wp_mail($to, $subject, $message, $headers);
        }
    }
}

// 初始化AJAX处理器
new NCEWPV_Ajax_Handler();