<?php
/**
 * 错误处理和日志记录类
 * 路径：includes/error-handler.php
 */

// 防止直接访问
if (!defined('ABSPATH')) {
    exit;
}

class NCEWPV_Error_Handler {
    
    private static $instance = null;
    private $log_table;
    
    private function __construct() {
        global $wpdb;
        $this->log_table = $wpdb->prefix . 'ncewpv_logs';
        
        // 注册错误处理钩子
        add_action('admin_notices', array($this, 'display_admin_errors'));
    }
    
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * 创建日志表（与安装脚本保持一致，双重保障）
     */
    public static function create_log_table() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();
        $table_name = $wpdb->prefix . 'ncewpv_logs';

        // 仅当表不存在时创建
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") != $table_name) {
            $sql = "CREATE TABLE $table_name (
                id mediumint(9) NOT NULL AUTO_INCREMENT,
                level varchar(20) NOT NULL,
                message text NOT NULL,
                context text,
                user_id bigint(20) NOT NULL DEFAULT 0,
                ip_address varchar(45) NOT NULL,
                user_agent text,
                created_at datetime DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (id),
                KEY level (level),
                KEY user_id (user_id),
                KEY created_at (created_at)
            ) $charset_collate;";

            require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
            dbDelta($sql);
        }
    }
    
    /**
     * 记录日志
     */
    public function log($level, $message, $context = array()) {
        global $wpdb;
        
        // 确保表存在
        if ($wpdb->get_var("SHOW TABLES LIKE '{$this->log_table}'") != $this->log_table) {
            self::create_log_table();
        }
        
        $data = array(
            'level' => strtoupper($level),
            'message' => $message,
            'context' => !empty($context) ? json_encode($context, JSON_UNESCAPED_UNICODE) : null,
            'user_id' => get_current_user_id(),
            'ip_address' => $this->get_user_ip(),
            'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : ''
        );
        
        $result = $wpdb->insert($this->log_table, $data);
        
        // 如果是严重错误，发送邮件通知
        if (in_array($level, array('error', 'critical'))) {
            $this->send_error_notification($level, $message, $context);
        }
        
        return $result;
    }
    
    /**
     * 获取用户IP地址
     */
    private function get_user_ip() {
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            return sanitize_text_field($_SERVER['HTTP_CLIENT_IP']);
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            return sanitize_text_field($_SERVER['HTTP_X_FORWARDED_FOR']);
        } else {
            return sanitize_text_field($_SERVER['REMOTE_ADDR']);
        }
    }
    
    /**
     * 发送错误通知邮件
     */
    private function send_error_notification($level, $message, $context = array()) {
        $admin_email = get_option('admin_email');
        if (empty($admin_email)) {
            return;
        }
        
        $subject = '[' . get_bloginfo('name') . '] 收费视频插件错误通知';
        
        $body = "
        <html>
        <head>
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: #d63638; color: white; padding: 20px; text-align: center; }
                .content { padding: 20px; background: #f9f9f9; }
                .footer { padding: 20px; text-align: center; font-size: 12px; color: #666; }
                .error-details { background: white; padding: 15px; border-left: 4px solid #d63638; margin: 15px 0; }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h1>系统错误通知</h1>
                </div>
                <div class='content'>
                    <p><strong>错误级别：</strong>{$level}</p>
                    <p><strong>错误信息：</strong>{$message}</p>
                    <p><strong>发生时间：</strong>" . current_time('mysql') . "</p>
                    
                    <div class='error-details'>
                        <strong>错误详情：</strong>
                        <pre>" . htmlspecialchars(print_r($context, true)) . "</pre>
                    </div>
                    
                    <p>您可以登录网站后台，在「系统日志」中查看更多详情。</p>
                </div>
                <div class='footer'>
                    此邮件由 " . get_bloginfo('name') . " 自动发送
                </div>
            </div>
        </body>
        </html>
        ";
        
        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . get_bloginfo('name') . ' <' . $admin_email . '>'
        );
        
        wp_mail($admin_email, $subject, $body, $headers);
    }
    
    /**
     * 获取日志统计
     */
    public function get_log_stats() {
        global $wpdb;
        
        return $wpdb->get_row("
            SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN level = 'CRITICAL' THEN 1 ELSE 0 END) as critical,
                SUM(CASE WHEN level = 'ERROR' THEN 1 ELSE 0 END) as errors,
                SUM(CASE WHEN level = 'WARNING' THEN 1 ELSE 0 END) as warnings,
                SUM(CASE WHEN level = 'INFO' THEN 1 ELSE 0 END) as info,
                SUM(CASE WHEN level = 'DEBUG' THEN 1 ELSE 0 END) as debug,
                MIN(created_at) as first_log,
                MAX(created_at) as last_log
            FROM {$this->log_table}
        ");
    }
    
    /**
     * 获取日志列表
     */
    public function get_logs($args = array()) {
        global $wpdb;
        
        $defaults = array(
            'level' => '',
            'user_id' => 0,
            'date_from' => '',
            'date_to' => '',
            'search' => '',
            'limit' => 50,
            'offset' => 0
        );
        
        $args = wp_parse_args($args, $defaults);
        
        $where = array('1=1');
        
        if (!empty($args['level'])) {
            $where[] = $wpdb->prepare("level = %s", $args['level']);
        }
        
        if ($args['user_id'] > 0) {
            $where[] = $wpdb->prepare("user_id = %d", $args['user_id']);
        }
        
        if (!empty($args['date_from'])) {
            $where[] = $wpdb->prepare("created_at >= %s", $args['date_from'] . ' 00:00:00');
        }
        
        if (!empty($args['date_to'])) {
            $where[] = $wpdb->prepare("created_at <= %s", $args['date_to'] . ' 23:59:59');
        }
        
        if (!empty($args['search'])) {
            $search = '%' . $wpdb->esc_like($args['search']) . '%';
            $where[] = $wpdb->prepare("(message LIKE %s OR context LIKE %s)", $search, $search);
        }
        
        $where_sql = '';
        if (!empty($where)) {
            $where_sql = "WHERE " . implode(' AND ', $where);
        }
        
        return $wpdb->get_results("
            SELECT * FROM {$this->log_table}
            {$where_sql}
            ORDER BY created_at DESC
            LIMIT {$args['limit']} OFFSET {$args['offset']}
        ");
    }
    
    /**
     * 清理日志
     */
    public function clear_logs($days = 30) {
        global $wpdb;
        
        if ($days > 0) {
            $date = date('Y-m-d H:i:s', strtotime("-$days days"));
            return $wpdb->query($wpdb->prepare(
                "DELETE FROM {$this->log_table} WHERE created_at < %s",
                $date
            ));
        } else {
            // 清空所有日志
            return $wpdb->query("TRUNCATE TABLE {$this->log_table}");
        }
    }
    
    /**
     * 显示后台错误提示
     */
    public function display_admin_errors() {
        // 可以在这里显示需要用户注意的错误信息
    }
}

// 初始化错误处理器（确保类加载后立即初始化）
if (!isset(NCEWPV_Error_Handler::$instance)) {
    NCEWPV_Error_Handler::get_instance();
}

/**
 * 便捷日志记录函数
 */
function ncewpv_log_error($message, $context = array()) {
    $handler = NCEWPV_Error_Handler::get_instance();
    return $handler->log('error', $message, $context);
}

function ncewpv_log_warning($message, $context = array()) {
    $handler = NCEWPV_Error_Handler::get_instance();
    return $handler->log('warning', $message, $context);
}

function ncewpv_log_info($message, $context = array()) {
    $handler = NCEWPV_Error_Handler::get_instance();
    return $handler->log('info', $message, $context);
}

function ncewpv_log_debug($message, $context = array()) {
    $handler = NCEWPV_Error_Handler::get_instance();
    return $handler->log('debug', $message, $context);
}