<?php
/**
 * 插件安装/卸载核心脚本
 * 正确路径：includes/ncewpv-install.php
 */

// 防止直接访问
if (!defined('ABSPATH')) {
    exit;
}

class NCEWPV_Install {
    /**
     * 安装核心逻辑（仅执行一次）
     */
    public static function run_install() {
        // 1. 创建数据库表
        self::create_log_table();
        self::create_orders_table();
        
        // 2. 创建必要的默认选项
        self::create_default_settings();
        
        // 3. 刷新重写规则
        flush_rewrite_rules();
        
        // 记录安装日志
        if (function_exists('ncewpv_log_info')) {
            ncewpv_log_info('插件安装完成', array(
                'version' => NCEWPV_PLUGIN_VERSION,
                'wordpress_version' => get_bloginfo('version'),
                'php_version' => PHP_VERSION
            ));
        }
    }

    /**
     * 卸载清理逻辑
     */
    public static function run_uninstall() {
        global $wpdb;
        
        // 获取插件设置
        $settings = get_option('ncewpv_settings', array());
        $delete_data = isset($settings['delete_data_on_uninstall']) ? $settings['delete_data_on_uninstall'] : 'no';
        
        if ($delete_data === 'yes') {
            // 1. 删除数据库表
            $wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}ncewpv_orders");
            $wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}ncewpv_logs");
            
            // 2. 删除插件选项
            delete_option('ncewpv_settings');
            delete_option('ncewpv_version');
            delete_transient('ncewpv_paypal_access_token');
            
            if (function_exists('ncewpv_log_info')) {
                ncewpv_log_info('插件卸载完成，数据已删除');
            }
        } else {
            if (function_exists('ncewpv_log_info')) {
                ncewpv_log_info('插件卸载完成，数据保留');
            }
        }
        
        // 3. 刷新重写规则
        flush_rewrite_rules();
    }

    /**
     * 创建日志表
     */
    private static function create_log_table() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();
        $table_name = $wpdb->prefix . 'ncewpv_logs';

        // 仅当表不存在时创建
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") != $table_name) {
            $sql = "CREATE TABLE $table_name (
                id mediumint(9) NOT NULL AUTO_INCREMENT,
                level varchar(20) NOT NULL,
                message text NOT NULL,
                context text,
                user_id bigint(20) NOT NULL DEFAULT 0,
                ip_address varchar(45) NOT NULL,
                user_agent text,
                created_at datetime DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (id),
                KEY level (level),
                KEY user_id (user_id),
                KEY created_at (created_at)
            ) $charset_collate;";

            require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
            dbDelta($sql);
        }
    }

    /**
     * 创建订单表
     */
    private static function create_orders_table() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();
        $table_name = $wpdb->prefix . 'ncewpv_orders';

        // 仅当表不存在时创建
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") != $table_name) {
            $sql = "CREATE TABLE $table_name (
                id mediumint(9) NOT NULL AUTO_INCREMENT,
                post_id bigint(20) NOT NULL,
                user_id bigint(20) NOT NULL,
                order_amount decimal(10,2) NOT NULL,
                paypal_order_id varchar(100),
                status varchar(20) DEFAULT 'pending',
                created_at datetime DEFAULT CURRENT_TIMESTAMP,
                updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (id),
                KEY post_id (post_id),
                KEY user_id (user_id),
                KEY paypal_order_id (paypal_order_id),
                KEY status (status),
                KEY created_at (created_at),
                KEY user_post (user_id, post_id),
                KEY status_created (status, created_at)
            ) $charset_collate;";

            require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
            dbDelta($sql);
        }
    }

    /**
     * 创建默认设置（避免因缺少选项导致报错）
     */
    private static function create_default_settings() {
        $default_settings = array(
            'environment' => 'sandbox',
            'client_id' => '',
            'client_secret' => '',
            'fee_rate' => 1.03,
            'fixed_fee' => 0,
            'currency' => 'USD',
            'delete_data_on_uninstall' => 'no',
            'enable_debug_log' => 'yes'
        );

        // 仅当选项不存在时添加
        if (!get_option('ncewpv_settings')) {
            add_option('ncewpv_settings', $default_settings);
        }
        
        // 保存插件版本
        update_option('ncewpv_version', NCEWPV_PLUGIN_VERSION);
    }
}