<?php
/**
 * PayPal支付处理器
 */

// 防止直接访问
if (!defined('ABSPATH')) {
    exit;
}

class NCEWPV_PayPal_Handler {
    
    private $client_id;
    private $client_secret;
    private $environment;
    private $base_url;
    
    public function __construct() {
        $settings = get_option('ncewpv_settings', array());
        
        $this->client_id = isset($settings['client_id']) ? $settings['client_id'] : '';
        $this->client_secret = isset($settings['client_secret']) ? $settings['client_secret'] : '';
        $this->environment = isset($settings['environment']) ? $settings['environment'] : 'sandbox';
        $this->base_url = $this->environment === 'live' 
            ? 'https://api.paypal.com' 
            : 'https://api.sandbox.paypal.com';
    }
    
    /**
     * 获取访问令牌
     */
    private function get_access_token() {
        // 检查是否有缓存的token
        $cached_token = get_transient('ncewpv_paypal_access_token');
        if ($cached_token) {
            return $cached_token;
        }

        // 验证凭证是否存在
        if (empty($this->client_id) || empty($this->client_secret)) {
            throw new Exception('Client ID或Secret未配置');
        }
        
        $url = $this->base_url . '/v1/oauth2/token';
        
        $args = array(
            'headers' => array(
                'Accept' => 'application/json',
                'Accept-Language' => 'en_US',
                'Authorization' => 'Basic ' . base64_encode($this->client_id . ':' . $this->client_secret),
            ),
            'body' => 'grant_type=client_credentials',
            'timeout' => 30
        );
        
        $response = wp_remote_post($url, $args);
        
        if (is_wp_error($response)) {
            ncewpv_log_error('获取PayPal访问令牌失败', array(
                'error' => $response->get_error_message(),
                'environment' => $this->environment,
                'url' => $url
            ));
            throw new Exception('连接PayPal失败: ' . $response->get_error_message() . '（请检查网络连接）');
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        $response_code = wp_remote_retrieve_response_code($response);
        
        if ($response_code !== 200) {
            ncewpv_log_error('获取PayPal访问令牌失败 - 状态码错误', array(
                'response_code' => $response_code,
                'response' => $body,
                'environment' => $this->environment
            ));
            $error_msg = isset($body['error_description']) ? $body['error_description'] : "HTTP状态码: $response_code";
            throw new Exception("认证失败: $error_msg");
        }
        
        if (isset($body['access_token'])) {
            // 缓存token，提前5分钟过期
            $expires_in = intval($body['expires_in']) - 300;
            set_transient('ncewpv_paypal_access_token', $body['access_token'], $expires_in);
            
            ncewpv_log_info('获取PayPal访问令牌成功', array(
                'environment' => $this->environment,
                'expires_in' => $expires_in
            ));
            
            return $body['access_token'];
        } else {
            ncewpv_log_error('获取PayPal访问令牌失败 - 响应错误', array(
                'response' => $body,
                'environment' => $this->environment
            ));
            throw new Exception('PayPal认证失败: ' . ($body['error_description'] ?? '未知错误'));
        }
    }
    
    /**
     * 创建PayPal订单
     */
    public function create_order($amount, $post_id) {
        $access_token = $this->get_access_token();
        
        $url = $this->base_url . '/v2/checkout/orders';
        
        $order_data = array(
            'intent' => 'CAPTURE',
            'purchase_units' => array(
                array(
                    'amount' => array(
                        'currency_code' => 'USD',
                        'value' => number_format($amount, 2, '.', '')
                    ),
                    'description' => '视频购买 - ' . get_the_title($post_id),
                    'custom_id' => 'post_' . $post_id
                )
            ),
            'application_context' => array(
                'brand_name' => get_bloginfo('name'),
                'landing_page' => 'BILLING',
                'user_action' => 'PAY_NOW',
                'return_url' => home_url('/'),
                'cancel_url' => home_url('/')
            )
        );
        
        $args = array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $access_token,
                'PayPal-Request-Id' => 'ncewpv_' . $post_id . '_' . time()
            ),
            'body' => json_encode($order_data),
            'timeout' => 30
        );
        
        $response = wp_remote_post($url, $args);
        
        if (is_wp_error($response)) {
            ncewpv_log_error('创建PayPal订单失败', array(
                'error' => $response->get_error_message(),
                'post_id' => $post_id,
                'amount' => $amount
            ));
            throw new Exception('创建订单失败: ' . $response->get_error_message());
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        $response_code = wp_remote_retrieve_response_code($response);
        
        if ($response_code !== 201) {
            ncewpv_log_error('创建PayPal订单失败 - 状态码错误', array(
                'response_code' => $response_code,
                'response' => $body,
                'post_id' => $post_id
            ));
            throw new Exception("创建订单失败: " . ($body['message'] ?? "HTTP状态码: $response_code"));
        }
        
        if (isset($body['id']) && $body['status'] === 'CREATED') {
            ncewpv_log_info('PayPal订单创建成功', array(
                'order_id' => $body['id'],
                'post_id' => $post_id,
                'amount' => $amount,
                'status' => $body['status']
            ));
            
            return $body;
        } else {
            ncewpv_log_error('创建PayPal订单失败 - 响应错误', array(
                'response' => $body,
                'post_id' => $post_id,
                'amount' => $amount
            ));
            throw new Exception('创建订单失败: ' . ($body['message'] ?? '未知错误'));
        }
    }
    
    /**
     * 捕获PayPal订单
     */
    public function capture_order($order_id, $post_id) {
        $access_token = $this->get_access_token();
        
        $url = $this->base_url . '/v2/checkout/orders/' . $order_id . '/capture';
        
        $args = array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $access_token,
                'Prefer' => 'return=representation'
            ),
            'timeout' => 30
        );
        
        $response = wp_remote_post($url, $args);
        
        if (is_wp_error($response)) {
            ncewpv_log_error('捕获PayPal订单失败', array(
                'error' => $response->get_error_message(),
                'order_id' => $order_id,
                'post_id' => $post_id
            ));
            throw new Exception('捕获订单失败: ' . $response->get_error_message());
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        $response_code = wp_remote_retrieve_response_code($response);
        
        if ($response_code !== 201) {
            ncewpv_log_error('捕获PayPal订单失败 - 状态码错误', array(
                'response_code' => $response_code,
                'response' => $body,
                'order_id' => $order_id
            ));
            throw new Exception("捕获订单失败: " . ($body['message'] ?? "HTTP状态码: $response_code"));
        }
        
        if (isset($body['status']) && $body['status'] === 'COMPLETED') {
            // 保存订单到数据库
            $this->save_order_to_db($body, $post_id);
            
            ncewpv_log_info('PayPal订单捕获成功', array(
                'order_id' => $order_id,
                'post_id' => $post_id,
                'capture_id' => $body['purchase_units'][0]['payments']['captures'][0]['id'],
                'amount' => $body['purchase_units'][0]['payments']['captures'][0]['amount']['value'],
                'status' => $body['status']
            ));
            
            return $body;
        } else {
            ncewpv_log_error('捕获PayPal订单失败 - 响应错误', array(
                'response' => $body,
                'order_id' => $order_id,
                'post_id' => $post_id
            ));
            throw new Exception('捕获订单失败: ' . ($body['message'] ?? '未知错误'));
        }
    }
    
    /**
     * 保存订单到数据库
     */
    private function save_order_to_db($order_data, $post_id) {
        global $wpdb;
        
        $user_id = get_current_user_id();
        $amount = $order_data['purchase_units'][0]['payments']['captures'][0]['amount']['value'];
        $paypal_order_id = $order_data['id'];
        
        $table_name = $wpdb->prefix . 'ncewpv_orders';
        
        $result = $wpdb->insert(
            $table_name,
            array(
                'post_id' => $post_id,
                'user_id' => $user_id,
                'order_amount' => $amount,
                'paypal_order_id' => $paypal_order_id,
                'status' => 'completed',
                'created_at' => current_time('mysql'),
                'updated_at' => current_time('mysql')
            ),
            array('%d', '%d', '%f', '%s', '%s', '%s', '%s')
        );
        
        if ($result === false) {
            ncewpv_log_error('保存订单到数据库失败', array(
                'post_id' => $post_id,
                'user_id' => $user_id,
                'amount' => $amount,
                'paypal_order_id' => $paypal_order_id,
                'db_error' => $wpdb->last_error
            ));
            throw new Exception('保存订单失败');
        }
        
        ncewpv_log_info('订单保存到数据库成功', array(
            'order_id' => $wpdb->insert_id,
            'post_id' => $post_id,
            'user_id' => $user_id,
            'amount' => $amount
        ));
        
        return $wpdb->insert_id;
    }
    
    /**
     * 测试PayPal连接
     */
    public function test_connection() {
        try {
            // 先清除缓存的token，确保获取新的token
            delete_transient('ncewpv_paypal_access_token');
            
            $access_token = $this->get_access_token();
            
            if ($access_token) {
                return array(
                    'success' => true,
                    'message' => 'PayPal连接测试成功'
                );
            } else {
                return array(
                    'success' => false,
                    'message' => '无法获取访问令牌'
                );
            }
        } catch (Exception $e) {
            return array(
                'success' => false,
                'message' => $e->getMessage()
            );
        }
    }
    
    /**
     * 获取订单详情
     */
    public function get_order_details($order_id) {
        $access_token = $this->get_access_token();
        
        $url = $this->base_url . '/v2/checkout/orders/' . $order_id;
        
        $args = array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $access_token,
                'Content-Type' => 'application/json'
            ),
            'timeout' => 30
        );
        
        $response = wp_remote_get($url, $args);
        
        if (is_wp_error($response)) {
            ncewpv_log_error('获取PayPal订单详情失败', array(
                'error' => $response->get_error_message(),
                'order_id' => $order_id
            ));
            throw new Exception('获取订单详情失败: ' . $response->get_error_message());
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($body['id'])) {
            return $body;
        } else {
            ncewpv_log_error('获取PayPal订单详情失败 - 响应错误', array(
                'response' => $body,
                'order_id' => $order_id
            ));
            throw new Exception('获取订单详情失败: ' . ($body['message'] ?? '未知错误'));
        }
    }
}