<?php
/**
 * WooCommerce集成
 */

// 防止直接访问
if (!defined('ABSPATH')) {
    exit;
}

class NCEWPV_WooCommerce_Integration {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function __construct() {
        // 确保在WooCommerce激活时加载
        if (class_exists('WooCommerce')) {
            $this->init();
        } else {
            // 如果WooCommerce未激活，监听其激活
            add_action('woocommerce_loaded', array($this, 'init'));
        }
    }
    
    public function init() {
        // 注册自定义端点
        add_action('init', array($this, 'add_videos_endpoint'), 10);
        
        // 查询变量
        add_filter('query_vars', array($this, 'add_query_vars'), 10);
        
        // 插入新菜单项 - 使用更高的优先级确保在其他项目之后添加
        add_filter('woocommerce_account_menu_items', array($this, 'add_videos_menu_item'), 20);
        
        // 设置端点内容
        add_action('woocommerce_account_my-videos_endpoint', array($this, 'videos_endpoint_content'));
        
        // 样式和脚本
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        
        // 刷新重写规则
        add_action('wp_loaded', array($this, 'flush_rewrite_rules'));
        
        // 调试信息
        add_action('wp_footer', array($this, 'debug_info'));
    }
    
    /**
     * 注册自定义端点
     */
    public function add_videos_endpoint() {
        add_rewrite_endpoint('my-videos', EP_ROOT | EP_PAGES);
    }
    
    /**
     * 添加查询变量
     */
    public function add_query_vars($vars) {
        $vars[] = 'my-videos';
        return $vars;
    }
    
    /**
     * 在"我的账户"菜单中添加视频菜单项
     */
    public function add_videos_menu_item($items) {
        // 在退出菜单前插入视频菜单项
        $new_items = array();
        
        $logout_item = array();
        if (isset($items['customer-logout'])) {
            $logout_item = array('customer-logout' => $items['customer-logout']);
            unset($items['customer-logout']);
        }
        
        // 添加视频菜单项
        $items['my-videos'] = __('我的视频', 'ncewpv');
        
        // 重新添加退出菜单
        if (!empty($logout_item)) {
            $items = array_merge($items, $logout_item);
        }
        
        return $items;
    }
    
    /**
     * 视频端点内容
     */
    public function videos_endpoint_content() {
        $user_id = get_current_user_id();
        
        if (!$user_id) {
            echo '<p>' . __('请先登录查看您的视频订单。', 'ncewpv') . '</p>';
            return;
        }
        
        $video_orders = $this->get_user_video_orders($user_id);
        
        ?>
        <div class="ncewpv-my-videos">
            <h2><?php _e('我的视频订单', 'ncewpv'); ?></h2>
            
            <?php if (empty($video_orders)): ?>
                <div class="woocommerce-message woocommerce-message--info woocommerce-Message woocommerce-Message--info">
                    <?php _e('您还没有购买过任何视频。', 'ncewpv'); ?>
                </div>
            <?php else: ?>
                <div class="ncewpv-video-orders">
                    <table class="woocommerce-orders-table woocommerce-MyAccount-orders shop_table shop_table_responsive my_account_orders account-orders-table">
                        <thead>
                            <tr>
                                <th class="woocommerce-orders-table__header woocommerce-orders-table__header-order-number">
                                    <span class="nobr"><?php _e('视频标题', 'ncewpv'); ?></span>
                                </th>
                                <th class="woocommerce-orders-table__header woocommerce-orders-table__header-order-date">
                                    <span class="nobr"><?php _e('购买日期', 'ncewpv'); ?></span>
                                </th>
                                <th class="woocommerce-orders-table__header woocommerce-orders-table__header-order-status">
                                    <span class="nobr"><?php _e('状态', 'ncewpv'); ?></span>
                                </th>
                                <th class="woocommerce-orders-table__header woocommerce-orders-table__header-order-total">
                                    <span class="nobr"><?php _e('金额', 'ncewpv'); ?></span>
                                </th>
                                <th class="woocommerce-orders-table__header woocommerce-orders-table__header-order-actions">
                                    <span class="nobr"><?php _e('操作', 'ncewpv'); ?></span>
                                </th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($video_orders as $order): ?>
                                <tr class="woocommerce-orders-table__row woocommerce-orders-table__row--status-<?php echo esc_attr($order->status); ?> order">
                                    <td class="woocommerce-orders-table__cell woocommerce-orders-table__cell-order-number" data-title="<?php _e('视频标题', 'ncewpv'); ?>">
                                        <?php echo $this->get_video_title_with_link($order->post_id); ?>
                                    </td>
                                    <td class="woocommerce-orders-table__cell woocommerce-orders-table__cell-order-date" data-title="<?php _e('购买日期', 'ncewpv'); ?>">
                                        <time datetime="<?php echo esc_attr($order->created_at); ?>">
                                            <?php echo date_i18n(get_option('date_format'), strtotime($order->created_at)); ?>
                                        </time>
                                    </td>
                                    <td class="woocommerce-orders-table__cell woocommerce-orders-table__cell-order-status" data-title="<?php _e('状态', 'ncewpv'); ?>">
                                        <?php echo $this->get_status_badge($order->status); ?>
                                    </td>
                                    <td class="woocommerce-orders-table__cell woocommerce-orders-table__cell-order-total" data-title="<?php _e('金额', 'ncewpv'); ?>">
                                        <?php echo wc_price($order->order_amount); ?>
                                    </td>
                                    <td class="woocommerce-orders-table__cell woocommerce-orders-table__cell-order-actions" data-title="<?php _e('操作', 'ncewpv'); ?>">
                                        <?php if ($order->status === 'completed'): ?>
                                            <a href="<?php echo get_permalink($order->post_id); ?>" class="woocommerce-button button view" target="_blank">
                                                <?php _e('观看视频', 'ncewpv'); ?>
                                            </a>
                                        <?php else: ?>
                                            <span class="woocommerce-button button disabled">
                                                <?php _e('等待支付', 'ncewpv'); ?>
                                            </span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                
                <!-- 视频统计 -->
                <div class="ncewpv-video-stats">
                    <div class="ncewpv-stat-card">
                        <h4><?php _e('总购买数', 'ncewpv'); ?></h4>
                        <span class="stat-number"><?php echo count($video_orders); ?></span>
                    </div>
                    <div class="ncewpv-stat-card">
                        <h4><?php _e('总消费额', 'ncewpv'); ?></h4>
                        <span class="stat-number">
                            <?php
                            $total_amount = array_sum(array_map(function($order) {
                                return floatval($order->order_amount);
                            }, $video_orders));
                            echo wc_price($total_amount);
                            ?>
                        </span>
                    </div>
                    <div class="ncewpv-stat-card">
                        <h4><?php _e('已完成订单', 'ncewpv'); ?></h4>
                        <span class="stat-number">
                            <?php
                            $completed_orders = array_filter($video_orders, function($order) {
                                return $order->status === 'completed';
                            });
                            echo count($completed_orders);
                            ?>
                        </span>
                    </div>
                </div>
                
            <?php endif; ?>
        </div>
        <?php
    }
    
    /**
     * 获取用户的视频订单
     */
    private function get_user_video_orders($user_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'ncewpv_orders';
        
        $orders = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table_name 
             WHERE user_id = %d 
             ORDER BY created_at DESC",
            $user_id
        ));
        
        return $orders;
    }
    
    /**
     * 获取带链接的视频标题
     */
    private function get_video_title_with_link($post_id) {
        $title = get_the_title($post_id);
        $permalink = get_permalink($post_id);
        
        if (empty($title)) {
            $title = __('视频已删除', 'ncewpv');
            return $title;
        }
        
        return '<a href="' . esc_url($permalink) . '" class="ncewpv-video-title">' . esc_html($title) . '</a>';
    }
    
    /**
     * 获取状态徽章
     */
    private function get_status_badge($status) {
        $status_labels = array(
            'pending' => array('label' => __('待支付', 'ncewpv'), 'class' => 'pending'),
            'completed' => array('label' => __('已完成', 'ncewpv'), 'class' => 'completed'),
            'cancelled' => array('label' => __('已取消', 'ncewpv'), 'class' => 'cancelled'),
            'refunded' => array('label' => __('已退款', 'ncewpv'), 'class' => 'refunded')
        );
        
        $status_info = isset($status_labels[$status]) ? $status_labels[$status] : $status_labels['pending'];
        
        return '<span class="ncewpv-status-badge ncewpv-status-' . $status_info['class'] . '">' . $status_info['label'] . '</span>';
    }
    
    /**
     * 加载样式和脚本
     */
    public function enqueue_scripts() {
        if (is_account_page()) {
            wp_enqueue_style(
                'ncewpv-woocommerce',
                NCEWPV_PLUGIN_URL . 'assets/css/woocommerce-integration.css',
                array(),
                NCEWPV_PLUGIN_VERSION
            );
        }
    }
    
    /**
     * 刷新重写规则
     */
    public function flush_rewrite_rules() {
        if (get_option('ncewpv_flush_rewrite_rules')) {
            flush_rewrite_rules();
            delete_option('ncewpv_flush_rewrite_rules');
            error_log('NCEWPV: 重写规则已刷新');
        }
    }
    
    /**
     * 调试信息
     */
    public function debug_info() {
        if (current_user_can('manage_options') && is_account_page()) {
            echo '<!-- NCEWPV Debug: WooCommerce Integration Loaded -->';
        }
    }
}

// 初始化WooCommerce集成
NCEWPV_WooCommerce_Integration::get_instance();