<?php
/**
 * Plugin Name: Namecheap EasyWP 收费视频
 * Plugin URI: https://yourwebsite.com/namecheap-easywp-paid-video
 * Description: 为EasyWP添加收费视频功能，支持PayPal支付
 * Version: 2.2.1
 * Author: life8882
 * Author URI: https://life8882.com
 * Text Domain: ncewpv
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * License: GPL v2 or later
 */

// 防止直接访问
if (!defined('ABSPATH')) {
    exit;
}

// 极简版本，移除所有重复检查
define('NCEWPV_PLUGIN_FILE', __FILE__);
define('NCEWPV_PLUGIN_URL', plugin_dir_url(__FILE__));
define('NCEWPV_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('NCEWPV_PLUGIN_VERSION', '2.2.1');

class Namecheap_EasyWP_Paid_Video {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('init', array($this, 'init'), 10);
        register_activation_hook(NCEWPV_PLUGIN_FILE, array($this, 'activate'));
        register_deactivation_hook(NCEWPV_PLUGIN_FILE, array($this, 'deactivate'));
    }
    
    public function activate() {
        // 简单的安装逻辑
        $this->create_tables();
        
        // 标记需要刷新重写规则
        update_option('ncewpv_flush_rewrite_rules', true);
        
        // 立即刷新重写规则
        flush_rewrite_rules();
        
        error_log('NCEWPV: 插件激活，重写规则已设置');
    }
    
    public function deactivate() {
        flush_rewrite_rules();
    }
    
    private function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // 创建订单表
        $orders_table = $wpdb->prefix . 'ncewpv_orders';
        $sql_orders = "CREATE TABLE $orders_table (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            post_id bigint(20) NOT NULL,
            user_id bigint(20) NOT NULL,
            order_amount decimal(10,2) NOT NULL,
            paypal_order_id varchar(100),
            status varchar(20) DEFAULT 'pending',
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id)
        ) $charset_collate;";
        
        // 创建日志表
        $logs_table = $wpdb->prefix . 'ncewpv_logs';
        $sql_logs = "CREATE TABLE $logs_table (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            level varchar(20) NOT NULL,
            message text NOT NULL,
            context text,
            user_id bigint(20) NOT NULL DEFAULT 0,
            ip_address varchar(45) NOT NULL,
            user_agent text,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        
        // 只创建不存在的表
        if ($wpdb->get_var("SHOW TABLES LIKE '$orders_table'") != $orders_table) {
            dbDelta($sql_orders);
        }
        
        if ($wpdb->get_var("SHOW TABLES LIKE '$logs_table'") != $logs_table) {
            dbDelta($sql_logs);
        }
        
        // 创建默认设置
        $default_settings = array(
            'environment' => 'sandbox',
            'client_id' => '',
            'client_secret' => '',
            'fee_rate' => 1.03,
            'fixed_fee' => 0
        );
        
        if (!get_option('ncewpv_settings')) {
            add_option('ncewpv_settings', $default_settings);
        }
    }
    
    public function init() {
        load_plugin_textdomain('ncewpv', false, dirname(plugin_basename(__FILE__)) . '/languages');
        
        $this->includes();
        $this->register_block_type();
        
        add_action('enqueue_block_editor_assets', array($this, 'enqueue_block_editor_assets'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_assets'));
        add_action('wp_head', array($this, 'add_paypal_sdk'));
    }
    
    private function includes() {
        $files = array(
            'includes/admin-settings.php',
            'includes/paypal-handler.php',
            'includes/ajax-handler.php',
            'includes/admin-orders.php',
            'includes/admin-logs.php',
            'includes/error-handler.php',
            'includes/woocommerce-integration.php', 
        );
        
        foreach ($files as $file) {
            $file_path = NCEWPV_PLUGIN_PATH . $file;
            if (file_exists($file_path)) {
                require_once $file_path;
            }
        }
    }
    
    private function register_block_type() {
        register_block_type('ncewpv/paid-video', array(
            'api_version' => 2,
            'title' => __('收费视频', 'ncewpv'),
            'description' => __('添加收费视频区块', 'ncewpv'),
            'category' => 'media',
            'icon' => 'video-alt3',
            'editor_script' => 'ncewpv-paid-video-block',
            'editor_style' => 'ncewpv-paid-video-block-editor',
            'style' => 'ncewpv-frontend',
            'render_callback' => array($this, 'render_paid_video_block'),
            'attributes' => array(
                'freeVideoUrl' => array('type' => 'string', 'default' => ''),
                'paidVideoUrl' => array('type' => 'string', 'default' => ''),
                'price' => array('type' => 'number', 'default' => 0),
                'previewLength' => array('type' => 'number', 'default' => 30)
            )
        ));
    }
    
    public function enqueue_block_editor_assets() {
        wp_enqueue_script(
            'ncewpv-paid-video-block',
            NCEWPV_PLUGIN_URL . 'assets/js/block-editor.js',
            array('wp-blocks', 'wp-element', 'wp-block-editor', 'wp-components', 'wp-i18n'),
            NCEWPV_PLUGIN_VERSION
        );
        
        wp_enqueue_style(
            'ncewpv-paid-video-block-editor',
            NCEWPV_PLUGIN_URL . 'assets/css/block-editor.css',
            array('wp-edit-blocks'),
            NCEWPV_PLUGIN_VERSION
        );
    }
    
    public function enqueue_frontend_assets() {
        if (has_block('ncewpv/paid-video')) {
            wp_enqueue_script(
                'ncewpv-frontend',
                NCEWPV_PLUGIN_URL . 'assets/js/frontend.js',
                array('jquery'),
                NCEWPV_PLUGIN_VERSION,
                true
            );
            
            wp_enqueue_style(
                'ncewpv-frontend',
                NCEWPV_PLUGIN_URL . 'assets/css/frontend.css',
                array(),
                NCEWPV_PLUGIN_VERSION
            );
            
            // 获取设置并传递给JavaScript
            $settings = get_option('ncewpv_settings', array());
            
            wp_localize_script('ncewpv-frontend', 'ncewpv_ajax', array(
                'ajaxurl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('ncewpv_ajax_nonce'),
                'settings' => $settings // 直接传递设置数据
            ));
        }
    }
    
    public function add_paypal_sdk() {
        if (has_block('ncewpv/paid-video')) {
            $settings = get_option('ncewpv_settings', array());
            $environment = isset($settings['environment']) ? $settings['environment'] : 'sandbox';
            $client_id = isset($settings['client_id']) ? $settings['client_id'] : '';
            
            if ($client_id) {
                $base_url = $environment === 'live'
                    ? 'https://www.paypal.com/sdk/js'
                    : 'https://www.sandbox.paypal.com/sdk/js';
                    
                $paypal_sdk_url = $base_url . '?client-id=' . $client_id . '&currency=USD&intent=capture';
                echo '<script src="' . esc_url($paypal_sdk_url) . '" data-ncewpv-paypal></script>';
            }
        }
    }
    
    public function render_paid_video_block($attributes, $content) {
        $attributes = wp_parse_args($attributes, array(
            'freeVideoUrl' => '',
            'paidVideoUrl' => '',
            'price' => 0,
            'previewLength' => 30
        ));
        
        $post_id = get_the_ID();
        $user_id = get_current_user_id();
        $has_purchased = $this->check_user_purchase($post_id, $user_id);
        
        // 计算最终价格
        $final_price = $this->calculate_final_price($attributes['price']);
        
        ob_start();
        ?>
        <div class="ncewpv-video-container" 
             data-post-id="<?php echo esc_attr($post_id); ?>"
             data-has-purchased="<?php echo $has_purchased ? '1' : '0'; ?>"
             data-attributes='<?php echo esc_attr(wp_json_encode($attributes)); ?>'>
            
            <?php if ($has_purchased && !empty($attributes['paidVideoUrl'])): ?>
                <div class="ncewpv-full-video">
                    <div class="ncewpv-video-wrapper">
                        <video controls playsinline width="100%">
                            <source src="<?php echo esc_url($attributes['paidVideoUrl']); ?>" type="video/mp4">
                            <?php _e('您的浏览器不支持视频播放', 'ncewpv'); ?>
                        </video>
                    </div>
                    <div class="ncewpv-purchase-success">
                        <h4><?php _e('已购买完整视频', 'ncewpv'); ?></h4>
                        <p><?php _e('您可以无限次观看此视频', 'ncewpv'); ?></p>
                    </div>
                </div>
            <?php elseif (!empty($attributes['freeVideoUrl'])): ?>
                <div class="ncewpv-preview-section">
                    <div class="ncewpv-free-video">
                        <h4><?php _e('免费预览', 'ncewpv'); ?></h4>
                        <div class="ncewpv-video-wrapper">
                            <video controls playsinline width="100%" data-preview-length="<?php echo esc_attr($attributes['previewLength']); ?>">
                                <source src="<?php echo esc_url($attributes['freeVideoUrl']); ?>" type="video/mp4">
                                <?php _e('您的浏览器不支持视频播放', 'ncewpv'); ?>
                            </video>
                        </div>
                        <p class="ncewpv-preview-info"><?php printf(__('预览时长: %s秒', 'ncewpv'), esc_html($attributes['previewLength'])); ?></p>
                    </div>
                    
                    <div class="ncewpv-purchase-section">
                        <div class="ncewpv-purchase-info">
                            <h4><?php _e('解锁完整视频', 'ncewpv'); ?></h4>
                            <div class="ncewpv-price-info">
                                <span class="ncewpv-original-price">
                                    <?php printf(__('原价: %s元', 'ncewpv'), esc_html($attributes['price'])); ?>
                                </span>
                                <span class="ncewpv-final-price">
                                    <?php printf(__('实付: %s元', 'ncewpv'), esc_html(number_format($final_price, 2))); ?>
                                </span>
                                <?php if ($final_price > $attributes['price']): ?>
                                    <small class="ncewpv-fee-notice"><?php _e('(含手续费)', 'ncewpv'); ?></small>
                                <?php endif; ?>
                            </div>
                            <button class="ncewpv-purchase-btn"><?php _e('立即购买完整视频', 'ncewpv'); ?></button>
                        </div>
                    </div>
                </div>
            <?php else: ?>
                <div class="ncewpv-video-error">
                    <p><?php _e('视频配置不完整，请检查区块设置', 'ncewpv'); ?></p>
                </div>
            <?php endif; ?>
        </div>
        <?php
        return ob_get_clean();
    }
    
    private function check_user_purchase($post_id, $user_id) {
        if (!$user_id || !$post_id) return false;
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'ncewpv_orders';
        
        $count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table_name WHERE post_id = %d AND user_id = %d AND status = 'completed'",
            $post_id, $user_id
        ));
        
        return $count > 0;
    }
    
    private function calculate_final_price($original_price) {
        $settings = get_option('ncewpv_settings', array());
        $fee_rate = isset($settings['fee_rate']) ? floatval($settings['fee_rate']) : 1.03;
        $fixed_fee = isset($settings['fixed_fee']) ? floatval($settings['fixed_fee']) : 0;
        
        $final_price = ($original_price * $fee_rate) + $fixed_fee;
        return round($final_price, 2);
    }
}

// 初始化插件
Namecheap_EasyWP_Paid_Video::get_instance();